package com.xebialabs.deployit.core.rest.api;

import javax.ws.rs.core.Response;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import com.xebialabs.deployit.core.api.UserProxy;
import com.xebialabs.deployit.core.api.dto.User;
import com.xebialabs.deployit.core.api.resteasy.http.tunnel.ResponseFactory;
import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource;
import com.xebialabs.deployit.security.JcrUser;
import com.xebialabs.deployit.security.UserService;
import com.xebialabs.deployit.security.permission.Permission;

@Controller
public class UserResource extends AbstractSecuredResource implements UserProxy {

	@Autowired
	UserService userService;

	@Override
	public Response create(final String username, final User user) {
		checkPermission(Permission.EDIT_SECURITY);
		sanityCheckUser(username, user);

		logger.debug("Creating user {}", username);
		userService.create(username, user.getPassword());
		JcrUser u = userService.read(username);
		logger.debug("Created user {}", u.getUsername());

		return ResponseFactory.created(new User(u.getUsername(), u.isAdmin())).build();
	}

	@Override
	public Response read(final String username) {
		checkPermission(Permission.EDIT_SECURITY);

		JcrUser u = userService.read(username);

		return ResponseFactory.ok(new User(u.getUsername(), u.isAdmin())).build();
	}

	@Override
	public Response modifyPassword(final String username, final User user) {
		checkPermission(Permission.EDIT_SECURITY);
		sanityCheckUser(username, user);

		userService.modifyPassword(username, user.getPassword());

		return ResponseFactory.noContent().build();
	}

	@Override
	public Response delete(final String username) {
		checkPermission(Permission.EDIT_SECURITY);

		userService.delete(username);

		return ResponseFactory.noContent().build();
	}

	private void sanityCheckUser(final String username, final User user) {
		if (user.getUsername() != null && !user.getUsername().equals(username)) {
			throw new IllegalArgumentException("Username in URL is " + username + " but username in POST data is " + user.getUsername());
		}
	}

    private static final Logger logger = LoggerFactory.getLogger(UserResource.class);
}
