package com.xebialabs.deployit.core.rest.api;

import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static java.lang.String.format;

import java.util.Calendar;

import javax.ws.rs.PathParam;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;

import com.google.common.base.Splitter;
import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.core.api.dto.FullTaskInfo;
import com.xebialabs.deployit.task.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import com.xebialabs.deployit.core.api.TaskProxy;
import com.xebialabs.deployit.core.api.dto.TaskInfo;
import com.xebialabs.deployit.core.api.resteasy.Date;
import com.xebialabs.deployit.core.api.resteasy.http.tunnel.ResponseFactory;
import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource;
import com.xebialabs.deployit.security.PermissionDeniedException;
import com.xebialabs.deployit.security.permission.Permission;

@SuppressWarnings("deprecation")
@Controller
public class TaskResource extends AbstractSecuredResource implements TaskProxy {

	@Autowired
	private DtoWriter dtoConverter;

	@Autowired
	private ExecutionEngine engine;

	@Autowired
	private TaskRegistry taskRegistry;

	@Autowired
	private TaskArchive taskArchive;

    @Override
	public Response start(final String taskId) {
		engine.execute(taskId);
		return ResponseFactory.accepted().build();
	}

    @Override
	public Response cancel(final String taskId) {
		engine.cancel(taskId);
		return ResponseFactory.status(Status.NO_CONTENT).build();
	}

    @Override
    public Response stop(final String taskId) {
        try {
            engine.stopExecution(taskId);
        } catch (IllegalStateException ise) {
            logger.error(format("Could not stop task %s", taskId), ise);
            return ResponseFactory.status(Response.Status.CONFLICT).entity(ise.getMessage()).build();
        }
        return ResponseFactory.accepted().build();
    }

    @Override
    public Response abort(final String taskId) {
        try {
            engine.abortExecution(taskId);
        } catch (IllegalStateException ise) {
            logger.error(format("Could not abort task %s", taskId), ise);
            return ResponseFactory.status(Response.Status.CONFLICT).entity(ise.getMessage()).build();
        }
        return ResponseFactory.accepted().build();
    }

    @Override
	public Response getTaskInfo(final String taskId) {
        Task runningTask = taskRegistry.getTask(taskId);
        TaskInfo taskInfoDto;
        if (runningTask != null) {
            taskInfoDto = dtoConverter.taskToDto(runningTask);
        } else {
            taskInfoDto = dtoConverter.archivedTaskToDto(taskArchive.getTask(taskId));
        }
        return ResponseFactory.ok(taskInfoDto).build();
	}

    @Override
    public Response assignTask(@PathParam("taskid") final String taskId, @PathParam("owner") final String owner) {
    	// Rules:
    	// - admin can reassign everyone's tasks
    	// - user can reassign his own tasks
        final Task task = taskRegistry.getTask(taskId);
	    checkArgument(task != null, "Could not find active task %s", taskId);

    	if (hasPermission(Permission.ADMIN)) {
    		taskRegistry.assignTask(taskId, owner);
    	} else if (hasPermission(Permission.TASK_ASSIGN)) {
    		taskRegistry.assignMyTask(taskId, owner);
    	} else {
    		throw PermissionDeniedException.withMessage("Neither ADMIN nor TASK_ASSIGN permissions are granted to you or you are not the owner of the task");
    	}

	    return ResponseFactory.ok().build();
    }
    
    @Override
    public Response toggleSkipSteps(@PathParam("taskid") final String taskId, @PathParam("steps") final String stepIds) {
	    final Task task = taskRegistry.getTask(taskId);
	    checkArgument(task != null, "Could not find active task %s", taskId);
	    if (!checkPermission(Permission.TASK_SKIPSTEP, task)) {
		    throw new Checks.IncorrectArgumentException("Can only skip steps on a deployment task");
	    }

        Iterable<String> taskStepIds = Splitter.on(",").split(stepIds);
        for (String stepId : taskStepIds) {
            final int id = Integer.parseInt(stepId);
            final TaskStep taskStep = task.getStep(id);
            if (taskStep.canSkip()) {
                logger.info("Skipping step :{}", taskStep.getDescription());
                taskStep.skip();
            } else if (taskStep.isSkipped()) {
                taskStep.unskip();
            }
        }

        return ResponseFactory.ok(dtoConverter.fullTaskToDto(task)).build();
    }

	private boolean checkPermission(Permission permission, Task task) {
		if (task instanceof DeploymentTask) {
			String env = ((DeploymentTask) task).getEnvironment();
			checkPermission(permission, env);
			return true;
		}
		return false;
	}

	@Override
	public Response moveStep(@PathParam("taskid") String taskId, @PathParam("stepNr") int stepNr, @PathParam("position") int newPosition) {
		final Task task = taskRegistry.getTask(taskId);
		checkArgument(task != null, "Could not find active task %s", taskId);
		if (!checkPermission(Permission.TASK_MOVESTEP, task)) {
			throw new Checks.IncorrectArgumentException("Can only move steps on a deployment task");
		}

		task.moveStep(stepNr, newPosition);

		return ResponseFactory.ok(dtoConverter.fullTaskToDto(task)).build();
	}

	@Override
	public Response getStepInfo(String taskId, int stepNr, Date ifModifiedSince) {
		Task runningTask = taskRegistry.getTask(taskId);
        TaskStepInfo step;
        if (runningTask != null) {
		    step = runningTask.getStep(stepNr);
        } else {
            DeploymentTaskInfo archivedTask = taskArchive.getTask(taskId);
            step = archivedTask.getStep(stepNr);
        }

        if (hasBeenModifiedSince(step, ifModifiedSince)) {
			return ResponseFactory.ok(dtoConverter.taskStepInfoToDto(stepNr, step)).lastModified(step.getLastModificationDate().getTime()).build();
		} else {
			return ResponseFactory.notModified().build();
		}
	}

	protected boolean hasBeenModifiedSince(TaskStepInfo step, Date ifModifiedSince) {
		if (ifModifiedSince == null) {
			return true;
		}

		Calendar lastModifiedCal = step.getLastModificationDate();
		lastModifiedCal.set(Calendar.MILLISECOND, 0); // set to 0 because time resolution is to 1 second in HTTP format
		return ifModifiedSince.getCalendar().before(lastModifiedCal);
	}


	/**
	 * Lists all unfinished tasks for the current user.
	 * 
	 * NOTE: this method is invoked from the GUI when restarting the UI after a crash. Therefore, it should only return
	 *       the tasks for the currently logged in user to prevent logging in as an administrator and seeing all tasks
	 *       in the system open in your GUI.
	 *       
	 *       If you do need the latter functionality, see getAllUnfinishedTasks().
	 */
	@Override
	public Response getUnfinishedTasks() {
		return ResponseFactory.ok(dtoConverter.tasksToDto(engine.getAllIncompleteTasksForCurrentUser())).build();
	}

	/**
	 * Lists all unfinished tasks in the system.
	 */
	@Override
	public Response getAllUnfinishedTasks() {
		checkPermission(Permission.ADMIN);
		return ResponseFactory.ok(dtoConverter.tasksToDto(engine.getAllIncompleteTasks())).build();
	}

	@Override
    public Response getStepsForTask(String taskId) {
		Task runningTask = taskRegistry.getTask(taskId);
        FullTaskInfo steps;
        if (runningTask != null) {
            steps = dtoConverter.fullTaskToDto(runningTask);
        } else {
            steps = dtoConverter.fullArchivedTaskToDto(taskArchive.getTask(taskId));
        }
		return ResponseFactory.ok(steps).build();
    }

	private static final Logger logger = LoggerFactory.getLogger(TaskResource.class);
}
