package com.xebialabs.deployit.core.rest.api;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.checks.Checks.checkNotNull;

import java.io.IOException;
import java.io.OutputStream;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.xebialabs.deployit.io.DerivedArtifactFile;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import com.google.common.collect.Maps;
import com.google.common.io.ByteStreams;
import com.google.common.io.OutputSupplier;
import com.xebialabs.deployit.core.api.dto.ArtifactAndData;
import com.xebialabs.deployit.core.api.dto.ConfigurationItemDto;
import com.xebialabs.deployit.core.api.dto.ConfigurationItemDtos;
import com.xebialabs.deployit.core.api.dto.RepositoryObject;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.DerivedArtifact;
import com.xebialabs.deployit.repository.RepositoryServiceHolder;
import com.xebialabs.deployit.repository.WorkDir;
import com.xebialabs.deployit.util.Either;
import com.xebialabs.deployit.util.PasswordObfuscator;
import com.xebialabs.overthere.local.LocalFile;

@Component
public class ConfigurationItemDtoReader {

	public <T extends ConfigurationItem> T read(ConfigurationItemDto dto) {
		return this.<T>read(dto, null);
	}

	public <T extends ConfigurationItem> T read(ConfigurationItemDto dto, WorkDir workDir) {
		return this.<T>read(dto, workDir, Maps.<String, Either<ConfigurationItemDto, ConfigurationItem>>newHashMap());
	}

	@SuppressWarnings("rawtypes")
    private <T extends ConfigurationItem> T read(ConfigurationItemDto dto, WorkDir workDir, Map<String, Either<ConfigurationItemDto, ConfigurationItem>> context) {
		checkNotNull(dto, "dto");
		checkNotNull(dto.getId(), "dto.id");
		Type type = Type.valueOf(dto.getType());
		checkArgument(type.exists(), "Type %s does not exist", type);

		Descriptor descriptor = type.getDescriptor();
		T item = descriptor.<T>newInstance();
		item.setId(dto.getId());

        context.put(item.getId(), Either.<ConfigurationItemDto, ConfigurationItem>right(item));

		copyValues(dto, item, descriptor, workDir, context);

		if (item instanceof DerivedArtifact && workDir != null) {
			logger.trace("Setting DerivedFile on {}", dto.getId());
			((DerivedArtifact) item).setFile(DerivedArtifactFile.create((DerivedArtifact) item));
		}

		return item;
	}

	public ConfigurationItem read(ArtifactAndData dto, WorkDir workDir) {
		checkNotNull(dto, "dto");

		Artifact read = (Artifact) read(dto.getArtifact(), workDir);
		read.setFile(readArtifactData(dto, workDir));
		return read;
	}

	private LocalFile readArtifactData(ArtifactAndData dto, WorkDir workDir) {
		checkArgument(dto.getArtifact().getFilename() != null, "The filename for the artifact should not be null");
		final LocalFile localFile = workDir.newFile(dto.getArtifact().getFilename());
		try {
			ByteStreams.write(dto.getData(), new OutputSupplier<OutputStream>() {
				@Override
				public OutputStream getOutput() throws IOException {
					return localFile.getOutputStream();
				}
			});
		} catch (IOException e) {
			throw new RuntimeException("Could not write Artifact data for " + dto.getArtifact().getId() + " to " + localFile);
		}
		return localFile;
	}

	private void copyValues(RepositoryObject src, ConfigurationItem dest, Descriptor descriptor, WorkDir workDir, Map<String, Either<ConfigurationItemDto, ConfigurationItem>> context) {
		for (PropertyDescriptor propertyDescriptor : descriptor.getPropertyDescriptors()) {
			Object o = src.getValues().get(propertyDescriptor.getName());
			if (o == null) continue;
			propertyDescriptor.set(dest, convertValueIfNecessary(o, propertyDescriptor, workDir, context));
		}
	}

	@SuppressWarnings("unchecked")
    private Object convertValueIfNecessary(Object dtoValue, PropertyDescriptor propertyDescriptor, WorkDir workDir, Map<String, Either<ConfigurationItemDto, ConfigurationItem>> context) {
		Object value = null;
		switch (propertyDescriptor.getKind()) {
			case STRING:
				if (propertyDescriptor.isPassword()) {
					value = PasswordObfuscator.ensureDecrypted((String) dtoValue);
					break;
				}
			case BOOLEAN:
			case INTEGER:
			case ENUM:
				value = dtoValue;
				break;
			case SET_OF_STRING:
				value = handleSetOfString(dtoValue);
				break;
			case SET_OF_CI:
				value = newHashSet(handleCollectionOfConfigurationItem(dtoValue, workDir, context));
				break;
			case LIST_OF_STRING:
				value = handleListOfString(dtoValue);
				break;
			case LIST_OF_CI:
				value = newArrayList(handleCollectionOfConfigurationItem(dtoValue, workDir, context));
				break;
			case CI:
				value = handleConfigurationItem((String) dtoValue, workDir, context);
				break;
			case MAP_STRING_STRING:
				//noinspection unchecked
				value = dtoValue != null ? newHashMap((Map<String, String>) dtoValue) : null;
				break;
		}

		return value;
	}

	@SuppressWarnings("unchecked")
    private Collection<ConfigurationItem> handleCollectionOfConfigurationItem(Object dtoValue, WorkDir workDir,
            Map<String, Either<ConfigurationItemDto, ConfigurationItem>> context) {
		Collection<ConfigurationItem> list = newArrayList();
	    for (String id : (Collection<String>) dtoValue) {
			list.add(handleConfigurationItem(id, workDir, context));
		}
		return list;
    }

	private ConfigurationItem handleConfigurationItem(String id, WorkDir workDir, Map<String, Either<ConfigurationItemDto, ConfigurationItem>> context) {
		if (context.containsKey(id)) {
			Either<ConfigurationItemDto, ConfigurationItem> either = context.get(id);
			if (either.getLeft() != null) {
				return read(either.getLeft(), workDir, context);
			} else {
				return either.getRight();
			}
		} else {
			ConfigurationItem ci = RepositoryServiceHolder.getRepositoryService().read(id, workDir);
			// register read ci.
			context.put(ci.getId(), Either.<ConfigurationItemDto, ConfigurationItem>right(ci));
			return ci;
		}
	}

	@SuppressWarnings({"unchecked"})
	private Set<String> handleSetOfString(Object dtoValue) {
		if (dtoValue instanceof Set) {
			return newHashSet((Set<String>) dtoValue);
		} else if (dtoValue instanceof Collection) {
			return newHashSet((Collection<String>) dtoValue);
		} else {
			throw new IllegalStateException("Did not get a Set of String or other Collection type, but a: " + dtoValue);
		}
	}
	
	@SuppressWarnings("unchecked")
    private List<String> handleListOfString(Object dtoValue) {
		if (dtoValue instanceof List) {
			return newArrayList((List<String>) dtoValue);
		} else if (dtoValue instanceof Collection) {
			return newArrayList((Collection<String>) dtoValue);
		} else {
			throw new IllegalStateException("Did not get a List of String or other Collection type, but a: " + dtoValue);
		}
	}

	public List<ConfigurationItem> read(ConfigurationItemDtos dtos) {
		return read(dtos.getObjects(), null);
	}

	public List<ConfigurationItem> read(List<ConfigurationItemDto> dtos) {
		return read(dtos, null);
	}
	public List<ConfigurationItem> read(List<ConfigurationItemDto> dtos, WorkDir workDir) {
		List<ConfigurationItem> cis = newArrayList();
		Map<String, Either<ConfigurationItemDto, ConfigurationItem>> context = buildContext(dtos);
		for (ConfigurationItemDto dto : dtos) {
			ConfigurationItem read = read(dto, workDir, context);
			cis.add(read);
		}
		return cis;
	}

	private Map<String, Either<ConfigurationItemDto, ConfigurationItem>> buildContext(List<ConfigurationItemDto> dtos) {
		HashMap<String, Either<ConfigurationItemDto, ConfigurationItem>> context = newHashMap();
		for (ConfigurationItemDto dto : dtos) {
			context.put(dto.getId(), Either.<ConfigurationItemDto, ConfigurationItem>left(dto));
		}
		return context;
	}

	private static final Logger logger = LoggerFactory.getLogger(ConfigurationItemDtoReader.class);
}
