package com.xebialabs.deployit.core.rest.api;

import static java.lang.String.format;

import java.io.File;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.List;

import javax.ws.rs.core.Response;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import com.xebialabs.deployit.core.api.ImportablePackageProxy;
import com.xebialabs.deployit.core.api.dto.FileUpload;
import com.xebialabs.deployit.core.api.dto.ImportablePackages;
import com.xebialabs.deployit.core.api.dto.RepositoryObject;
import com.xebialabs.deployit.core.api.resteasy.http.tunnel.ResponseFactory;
import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.server.api.importer.ImportSource;
import com.xebialabs.deployit.service.importer.ImporterException;
import com.xebialabs.deployit.service.importer.ImporterService;
import com.xebialabs.deployit.service.importer.source.FileSource;
import com.xebialabs.deployit.service.importer.source.InputStreamSource;
import com.xebialabs.deployit.service.importer.source.UrlSource;


/**
 * I am A REST resource that exposes functionality on the Deployable packages.
 * <p/>
 * Functionality includes: - Listing all Deployment packages in the importablePackages directory on the server.
 */
@Controller
public class ImportablePackageResource extends AbstractSecuredResource implements ImportablePackageProxy {

	@Autowired
	private ImporterService importerService;

	@Autowired
	private RepositoryService repositoryService;

    @Autowired
    private ConfigurationItemDtoWriter dtoWriter;

	public Response list() {
		List<String> files = importerService.listPackages();
		return ResponseFactory.ok(new ImportablePackages(files)).build();
	}

	public Response importPackage(String file) {
		if (!importerService.getImportablePackageDirectory().exists()) {
			throw new ImporterException(format("Directory \'%s\' to import packages from does not exist", importerService.getImportablePackageDirectory().getAbsolutePath()));
		}
		FileSource fileSource = new FileSource(new File(importerService.getImportablePackageDirectory(), file), false);
		return ResponseFactory.created(doImport(fileSource)).build();	
	}

	@Override
    public Response importPackageFromUrl(String url) {
	    try {
		    UrlSource urlSource = new UrlSource(new URL(url));
		    return ResponseFactory.created(doImport(urlSource)).build();
	    } catch (MalformedURLException e) {
		    throw new RuntimeException(e);
	    }
    }

    @Override
	public Response importUploadedPackage(String file, FileUpload form) {
		InputStream uploadedPackage = form.getFileData();
	    InputStreamSource byteArraySource = new InputStreamSource(file, uploadedPackage);
		return ResponseFactory.created(doImport(byteArraySource)).build();
	}

	private RepositoryObject doImport(ImportSource source) {
		String importedApplication = importerService.importPackage(source);
		ConfigurationItem read = repositoryService.read(importedApplication);
		return dtoWriter.write(read);
	}
}
