package com.xebialabs.deployit.core.rest.api.reports;

import static com.xebialabs.deployit.task.Task.State.DONE;

import java.util.List;
import java.util.Map;

import com.google.common.collect.Maps;
import com.xebialabs.deployit.core.api.dto.ConfigurationItemId;
import com.xebialabs.deployit.core.api.dto.Report;
import com.xebialabs.deployit.core.api.resteasy.Date;
import com.xebialabs.deployit.task.ArchivedTaskSearchParameters;
import com.xebialabs.deployit.task.DeploymentTaskInfo;
import com.xebialabs.deployit.task.TaskArchive;
import com.xebialabs.deployit.task.jcrarchive.JcrTaskArchive.TaskCallback;

public class DeploymentsStateBreakdownReport {

	private TaskArchive taskArchive;

	public DeploymentsStateBreakdownReport(TaskArchive taskArchive) {
		this.taskArchive = taskArchive;
	}

	public Report report(final List<ConfigurationItemId> filterCriteria, final Date startDate, final Date endDate, final ReportFilterType filterType) {
		final Map<String, TaskStateData> tasksMap = Maps.newHashMap();

		ArchivedTaskSearchParameters searchTasks = new ArchivedTaskSearchParameters();
		searchTasks.createdBetween(startDate.getCalendar(), endDate.getCalendar());

		for (ConfigurationItemId appOrEnvName : filterCriteria) {
			switch (filterType) {
			case APPLICATION:
				searchTasks.forApplication(appOrEnvName.getId());
				break;
			case ENVIRONMENT:
				searchTasks.toEnvironment(appOrEnvName.getId());
				break;
			default:
				throw new UnsupportedOperationException("Report filter type, " + filterType + " is not supported");
			}

			taskArchive.searchTasksWithoutLoadingSteps(searchTasks, new TaskCallback() {
				public void doWithTask(DeploymentTaskInfo task) {
					createFilteredTasksMap(tasksMap, task, filterType);

				}
			});
		}

		for (ConfigurationItemId entityId : filterCriteria) {
			if (!tasksMap.containsKey(entityId.getId())) {
				tasksMap.put(entityId.getId(), new TaskStateData());
			}
		}
		return generateReportDataFromTaskMap(tasksMap);
	}

	private Report generateReportDataFromTaskMap(Map<String, TaskStateData> tasksMap) {
		final Report report = new Report();
		for (Map.Entry<String, TaskStateData> entry : tasksMap.entrySet()) {
			final Report.ReportLine reportLine = report.addLine();
			reportLine.addValue("entityName", entry.getKey());
			reportLine.addValue("noOfSuccessfulDeployments", entry.getValue().getNoOfSuccessfulDeployments().toString());
		}
		return report;
	}

	private void createFilteredTasksMap(Map<String, TaskStateData> tasksMap, DeploymentTaskInfo task, ReportFilterType filterType) {
		String entityId = getEntityIdFromDeploymentTask(task, filterType);
		if (task.getState().equals(DONE)) {
			if (!tasksMap.containsKey(entityId)) {
				TaskStateData taskStateData = new TaskStateData();
				taskStateData.getNoOfSuccessfulDeployments().incrementAndGet();
				tasksMap.put(entityId, taskStateData);
			} else {
				tasksMap.get(entityId).getNoOfSuccessfulDeployments().incrementAndGet();
			}
		}
	}

	private String getEntityIdFromDeploymentTask(DeploymentTaskInfo inputTask, ReportFilterType filterType) {
		String entityId = "";
		switch (filterType) {
		case APPLICATION:
			entityId = inputTask.getApplicationName();
			break;
		case ENVIRONMENT:
			entityId = inputTask.getEnvironment();
			break;
		default:
			throw new UnsupportedOperationException("Report filter type, " + filterType + " is not supported");
		}

		return entityId;
	}

}
