package com.xebialabs.deployit.core.rest.api;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.Collections2;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.Lists;
import com.xebialabs.deployit.core.api.QueryProxy;
import com.xebialabs.deployit.core.api.dto.*;
import com.xebialabs.deployit.core.api.resteasy.Date;
import com.xebialabs.deployit.core.api.resteasy.http.tunnel.ResponseFactory;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.repository.ConfigurationItemData;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.repository.SearchParameters;
import com.xebialabs.deployit.security.permission.Permission;
import com.xebialabs.deployit.service.comparison.Comparator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Response;
import java.util.Collection;
import java.util.List;

import static com.google.common.collect.Collections2.filter;
import static com.google.common.collect.Lists.*;
import static com.xebialabs.deployit.core.rest.api.DtoReader.ciDataToCiId;
import static com.xebialabs.deployit.core.rest.api.DtoReader.ciIdToCiData;
import static com.xebialabs.deployit.core.rest.api.SearchParameterFactory.createSearchParam;

/**
 */
@Controller
public class QueryResource implements QueryProxy {

	public static final Predicate<ConfigurationItemData> hasReadRight = new Predicate<ConfigurationItemData>() {
		@Override
		public boolean apply(ConfigurationItemData input) {
			return Permission.READ.getPermissionHandler().hasPermission(input.getId());
		}
	};

	@Autowired
	private RepositoryService repositoryService;

	@Autowired
	private ConfigurationItemDtoWriter dtoWriter;

	@Autowired
    private Comparator comparator;

	@Override
	public Response list(@QueryParam("typeName") final String typeName, @QueryParam("archetype") final boolean archetype, @QueryParam("page") final long page,
	        @QueryParam("resultsPerPage") final long resultPerPage, @QueryParam("property") final List<PropertyValue> values, final String parent, final String id) {
		SearchParameters searchParam = createSearchParam(typeName, page, resultPerPage, values, parent, id);
		ConfigurationItemIds dto = new ConfigurationItemIds();
		Collection<ConfigurationItemId> filter = Collections2.transform(filter(repositoryService.list(searchParam), hasReadRight), ciDataToCiId);
		dto.getConfigurationItemIds().addAll(filter);
		return ResponseFactory.ok(dto).build();
	}

	@Override
	public Response readMultiple(ConfigurationItemIds configurationItemIds) {
		final ConfigurationItemDtos result = new ConfigurationItemDtos();
		Collection<ConfigurationItemData> datas = filter(transform(configurationItemIds.getConfigurationItemIds(), ciIdToCiData), hasReadRight);
		for (ConfigurationItemData data : datas) {
			try {
				final ConfigurationItem entity = repositoryService.read(data.getId());
				result.add(dtoWriter.write(entity));
			} catch (RuntimeException re) {
				logger.error("Could not read configuration item [{}]", data);
				logger.error("Exception was: ", re);
			}
		}
		return ResponseFactory.ok(result).build();
	}

    @Override
    public Response compare(final String reference, final List<String> ids) {
        ConfigurationItem referenceEntity = repositoryService.read(reference);

        final ListMultimap<String,String> multimap = comparator.compare(referenceEntity, Lists.transform(ids, new Function<String, ConfigurationItem>() {
	        @Override
	        public ConfigurationItem apply(String input) {
		        return repositoryService.read(input);
	        }
        }));

        return ResponseFactory.ok(Comparison.from(multimap)).build();
    }


	@Override
	public Response search(@QueryParam("typeName") String typeName, @QueryParam("before") Date date) {
		logger.info("Searching for all [{}] with last modified before [{}]", typeName, date);
		SearchParameters searchParam = createSearchParam(typeName, 0, -1, Lists.<PropertyValue>newArrayList());
		searchParam.setBefore(date.getCalendar());
		List<ConfigurationItemData> ids = repositoryService.list(searchParam);
		logger.debug("Found ids: {}", ids);
		return ResponseFactory.ok(new ConfigurationItemIds(transform(ids, ciDataToCiId))).build();
	}
	
	private static final Logger logger = LoggerFactory.getLogger(QueryResource.class);
}
