package com.xebialabs.deployit.core.rest.api;


import com.google.common.base.Function;
import com.google.common.collect.Collections2;
import com.google.common.collect.Iterables;
import com.google.common.collect.Multimap;
import com.xebialabs.deployit.core.api.dto.*;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.MethodDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.security.permission.Permission;
import com.xebialabs.deployit.task.DeploymentTask;
import com.xebialabs.deployit.task.DeploymentTaskInfo;
import com.xebialabs.deployit.task.Task;
import com.xebialabs.deployit.task.TaskStepInfo;
import org.springframework.stereotype.Component;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Lists.transform;
import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.checks.Checks.checkNotNull;

@Component
public class DtoWriter {

    public ConfigurationItemDescriptorDto descriptorToDto(final Descriptor descriptor) {
        checkNotNull(descriptor, "descriptor");

        final ConfigurationItemDescriptorDto dto = new ConfigurationItemDescriptorDto();
        dto.setType(descriptor.getType().toString());
        Function<Type, String> typeToString = new Function<Type, String>() {
            public String apply(Type from) {
                return from.toString();
            }
        };
        dto.setInterfaces(newHashSet(Collections2.transform(descriptor.getInterfaces(), typeToString)));
        dto.setSuperClasses(newArrayList(Collections2.transform(descriptor.getSuperClasses(), typeToString)));
        Type deployableType = descriptor.getDeployableType();
        dto.setMappingSourceType(deployableType != null ? deployableType.toString() : "");
        Type containerType = descriptor.getContainerType();
        dto.setMappingTargetType(containerType != null ? containerType.toString() : "");
        dto.setDescription(descriptor.getDescription());
        dto.setRootName(descriptor.getRoot().getRootNodeName());
        for (PropertyDescriptor propertyDescriptor : descriptor.getPropertyDescriptors()) {
            if (!propertyDescriptor.isHidden()) {
                dto.addPropertyDescriptor(propertyDescriptorToDto(propertyDescriptor));
            }
        }

	    for (MethodDescriptor controlTask : descriptor.getControlTasks()) {
		    dto.addControlTask(methodDescriptorToDto(controlTask));
	    }

        return dto;
    }

	private ConfigurationItemMethodDescriptorDto methodDescriptorToDto(MethodDescriptor controlTask) {
		ConfigurationItemMethodDescriptorDto configurationItemMethodDescriptorDto = new ConfigurationItemMethodDescriptorDto();
		configurationItemMethodDescriptorDto.setDescription(controlTask.getDescription());
		configurationItemMethodDescriptorDto.setName(controlTask.getName());
		return configurationItemMethodDescriptorDto;
	}

	private ConfigurationItemPropertyDescriptorDto propertyDescriptorToDto(final PropertyDescriptor propertyDescriptor) {
        checkNotNull(propertyDescriptor, "propertydescriptor");

        final ConfigurationItemPropertyDescriptorDto dto = new ConfigurationItemPropertyDescriptorDto();
        dto.setName(propertyDescriptor.getName());
        dto.setLabel(propertyDescriptor.getLabel());
        dto.setType(ConfigurationItemPropertyKind.valueOf(propertyDescriptor.getKind().toString()));
        if (propertyDescriptor.getEnumValues() != null) {
            dto.setEnumValues(propertyDescriptor.getEnumValues().toArray(new String[propertyDescriptor.getEnumValues().size()]));
        }
        Type referencedType = propertyDescriptor.getReferencedType();
        dto.setCollectionMemberClassname(referencedType != null ? referencedType.toString() : "");
        dto.setPropertyClassname(""); // FIXME: for what was this used?
        dto.setRequired(propertyDescriptor.isRequired());
        dto.setEditable(true); // no read-only properties anymore
        dto.setPassword(propertyDescriptor.isPassword());
        dto.setIdentifying(false); // no identifying parameters anymore
        dto.setDiscoveryParam(false); // no discovery parameters anymore
        dto.setDiscoveryRequired(false); // no discovery parameters anymore
        dto.setCategory(propertyDescriptor.getCategory());
        dto.setSize(propertyDescriptor.getSize().toString());
        Object defaultValue = propertyDescriptor.getDefaultValue();
        dto.setDefaultValue(defaultValue == null ? null : defaultValue.toString());
        dto.setDescription(propertyDescriptor.getDescription());
        dto.setAsContainment(propertyDescriptor.isAsContainment());
        return dto;
    }

    public TaskInfos tasksToDto(final List<Task> tasks) {
        TaskInfos taskInfos = new TaskInfos();
        for (Task eachTask : tasks) {
            taskInfos.add(taskToDto(eachTask));
        }
        return taskInfos;
    }

   public TaskInfo taskToDto(Task task) {
       return taskToDto(task, new TaskInfo());
    }

    public FullTaskInfo fullTaskToDto(final Task task) {
        int stepNr = 1;
        FullTaskInfo fti = taskToDto(task, new FullTaskInfo());
        for (TaskStepInfo taskStepInfo : task.getSteps()){
            fti.addStep(taskStepInfoToDto(stepNr++, taskStepInfo));
        }
        return fti;
    }

    private <T extends TaskInfo> T taskToDto(Task task, T ti) {
        checkNotNull(task, "task");
        checkNotNull(task.getId(), "id");

        ti.setId(task.getId());
        ti.setLabel(task.getLabel());
        ti.setState(task.getState().name());
        ti.setStartDate(task.getStartDate());
        ti.setCompletionDate(task.getCompletionDate());
        ti.setNrOfSteps(task.getNrOfSteps());
        ti.setCurrentStepNr(task.getCurrentStepNr());
        ti.setUser(task.getOwner());

        if (task instanceof DeploymentTask) {
            DeploymentTask dt = (DeploymentTask) task;
            ti.setApplication(dt.getApplicationName());
            ti.setVersion(dt.getApplicationVersion());
            ti.setEnvironment(dt.getEnvironment());
            ti.setFailureCount(dt.getFailureCount());
        }

        return ti;
    }

    public StepInfo taskStepInfoToDto(final int stepNr, final TaskStepInfo step) {
        checkNotNull(step, "step");

        StepInfo si = new StepInfo();
        si.setNr(stepNr);
        si.setDescription(step.getDescription());
        si.setState(step.getState().name());
        si.setStartDate(step.getStartDate());
        si.setCompletionDate(step.getCompletionDate());
        si.setLog(step.getLog());

        return si;
    }

    public TaskInfo archivedTaskToDto(DeploymentTaskInfo deploymentTask) {
        checkNotNull(deploymentTask, "deploymentTask");
        
        return populateTaskInfo(deploymentTask, new TaskInfo());
    }

	public FullTaskInfo fullArchivedTaskToDto(final DeploymentTaskInfo deploymentTaskInfo) {
		checkNotNull(deploymentTaskInfo, "deploymentTaskInfo");

		int stepNr = 1;
		FullTaskInfo fullTaskInfo = populateTaskInfo(deploymentTaskInfo, new FullTaskInfo());
        for (TaskStepInfo taskStepInfo : deploymentTaskInfo.getSteps()){
        	fullTaskInfo.addStep(taskStepInfoToDto(stepNr++, taskStepInfo));
        }
		return fullTaskInfo;
	}

    private <T extends TaskInfo> T populateTaskInfo(final DeploymentTaskInfo deploymentTask, final T taskInfo) {
        checkNotNull(deploymentTask, "deploymentTask");
        checkNotNull(taskInfo,"taskInfo");

        taskInfo.setId(deploymentTask.getId());
        taskInfo.setLabel(String.format("Deployment of package:%s version:%s to env:%s", deploymentTask.getApplicationName(), deploymentTask.getApplicationVersion(), deploymentTask.getEnvironment()));
        taskInfo.setState(deploymentTask.getState().name());
        taskInfo.setStartDate(deploymentTask.getStartDate());
        taskInfo.setCompletionDate(deploymentTask.getCompletionDate());
        taskInfo.setNrOfSteps(deploymentTask.getSteps().size());
        taskInfo.setCurrentStepNr(deploymentTask.getCurrentStepNr());
        taskInfo.setUser(deploymentTask.getOwner());
        taskInfo.setEnvironment(deploymentTask.getEnvironment());
        taskInfo.setApplication(deploymentTask.getApplicationName());
        taskInfo.setVersion(deploymentTask.getApplicationVersion());
        return taskInfo;
    }

	public List<Role> writeRoles(List<com.xebialabs.deployit.security.Role> roles) {
		return newArrayList(transform(roles, new Function<com.xebialabs.deployit.security.Role, Role>() {
			public Role apply(com.xebialabs.deployit.security.Role input) {
				return writeRole(input);
			}
		}));
	}

	private Role writeRole(com.xebialabs.deployit.security.Role role) {
		return new Role(role.getId(), role.getName());
	}

	public List<RoleAssignment> writeRoleAssignments(List<com.xebialabs.deployit.security.Role> roles) {
		List<RoleAssignment> assignments = newArrayList();
		for (com.xebialabs.deployit.security.Role role : roles) {
			assignments.add(new RoleAssignment(writeRole(role), newArrayList(role.getPrincipalsAssigned())));
		}
		return assignments;
	}

	public List<RolePermission> writeRolePermissions(Multimap<com.xebialabs.deployit.security.Role, Permission> multimap) {
		List<RolePermission> permissions = newArrayList();
		for (com.xebialabs.deployit.security.Role role : multimap.keySet()) {
			permissions.add(new RolePermission(writeRole(role), writePermissions(multimap.get(role))));
		}

		return permissions;
	}

	private List<String> writePermissions(Collection<Permission> permissions) {
		return newArrayList(Iterables.transform(permissions, new Function<Permission, String>() {
			public String apply(Permission input) {
				return input.getPermissionName();
			}
		}));
	}

	public List<com.xebialabs.deployit.core.api.dto.Permission> writeReferenceDataPermissions(Permission[] values) {
		return newArrayList(Iterables.transform(Arrays.asList(values), new Function<Permission, com.xebialabs.deployit.core.api.dto.Permission>() {
			public com.xebialabs.deployit.core.api.dto.Permission apply(Permission input) {
				String rootNodeName = input.getRoot() != null ? input.getRoot().getRootNodeName() : null;
				return new com.xebialabs.deployit.core.api.dto.Permission(input.getPermissionName(), input.getLevel().name(), rootNodeName);
			}
		}));
	}
}
