package com.xebialabs.deployit.core.rest.resteasy;

import com.google.common.io.Closeables;
import org.apache.james.mime4j.storage.Storage;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.util.Set;

import static com.google.common.collect.Sets.newHashSet;

public class TemporaryFiles {

	private static final ThreadLocal<Set<TempFileStorage>> tempFiles = new ThreadLocal<Set<TempFileStorage>>() {
		@Override
		protected Set<TempFileStorage> initialValue() {
			return newHashSet();
		}
	};

	public static void clear() {
		// defensive copy so that we can delete from the collection.
		for (TempFileStorage tempFileStorage : newHashSet(tempFiles.get())) {
			if (tempFileStorage.deleteWithResponse()) {
				tempFiles.get().remove(tempFileStorage);
			}
		}
	}

	/**
	 * Adaption of the standard TempFileStorage found in Mime4j's TemporaryFileStorageProvider.
	 * It keeps a reference to all the streams that have been opened on the file, so that they
	 * can be closed before deleting the file.
	 */
	public static final class TempFileStorage implements Storage {

		private File file;

		private Set<InputStream> streamsOpened = newHashSet();
		
		public TempFileStorage(File file) {
			this.file = file;
			// Register me for deletion upon thread exit.
			TemporaryFiles.tempFiles.get().add(this);
			logger.debug("Registered temporary file: {}", file);
		}

		public void delete() {
			if (file == null) return;

			logger.debug("Deleting temporary file: {}", file);
			
			for (InputStream inputStream : streamsOpened) {
				Closeables.closeQuietly(inputStream);
			}
			streamsOpened.clear();
			
			if (file != null) {
				if (file.delete()) {
					file = null;
				}
			}
		}
		
		private boolean deleteWithResponse() {
			delete();
			return file == null;
		}

		public InputStream getInputStream() throws IOException {
			if (file == null)
				throw new IllegalStateException("storage has been deleted");

			BufferedInputStream bufferedInputStream = new BufferedInputStream(new FileInputStream(file));
			streamsOpened.add(bufferedInputStream);
			return bufferedInputStream;
		}
	}
	
	private static final Logger logger = LoggerFactory.getLogger(TemporaryFiles.class);
}
