package com.xebialabs.deployit.core.rest.api;

import com.google.common.base.Function;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.Lists;
import com.xebialabs.deployit.core.api.QueryProxy;
import com.xebialabs.deployit.core.api.dto.Comparison;
import com.xebialabs.deployit.core.api.dto.PropertyValue;
import com.xebialabs.deployit.core.api.dto.RepositoryObjectIds;
import com.xebialabs.deployit.core.api.dto.ConfigurationItemDtos;
import com.xebialabs.deployit.core.api.resteasy.Date;
import com.xebialabs.deployit.core.api.resteasy.http.tunnel.ResponseFactory;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.repository.SearchParameters;
import com.xebialabs.deployit.service.comparison.Comparator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Response;
import java.util.List;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.core.rest.api.SearchParameterFactory.createSearchParam;

/**
 */
@Controller
public class QueryResource implements QueryProxy {

	@Autowired
	private RepositoryService repositoryService;

	@Autowired
	private ConfigurationItemDtoWriter dtoWriter;

	@Autowired
    private Comparator comparator;

	@Override
	public Response list(@QueryParam("typeName") final String typeName, @QueryParam("archetype") final boolean archetype, @QueryParam("page") final long page,
	        @QueryParam("resultsPerPage") final long resultPerPage, @QueryParam("property") final List<PropertyValue> values, final String parent, final String id) {
		SearchParameters searchParam = createSearchParam(typeName, page, resultPerPage, values, parent, id);
		RepositoryObjectIds dto = new RepositoryObjectIds();
		dto.setRepositoryEntityIds(repositoryService.list(searchParam));
		return ResponseFactory.ok(dto).build();
	}

	@Override
	public Response readMultiple(RepositoryObjectIds repositoryObjectIds) {
		final ConfigurationItemDtos result = new ConfigurationItemDtos();
		for (String repositoryObjectId : repositoryObjectIds.getRepositoryEntityIds()) {
			final ConfigurationItem entity = repositoryService.read(repositoryObjectId);
			result.add(dtoWriter.write(entity));
		}
		return ResponseFactory.ok(result).build();
	}

    @Override
    public Response compare(final String reference, final List<String> ids) {
        ConfigurationItem referenceEntity = repositoryService.read(reference);

        final ListMultimap<String,String> multimap = comparator.compare(referenceEntity, Lists.transform(ids, new Function<String, ConfigurationItem>() {
	        @Override
	        public ConfigurationItem apply(String input) {
		        return repositoryService.read(input);
	        }
        }));

        return ResponseFactory.ok(Comparison.from(multimap)).build();
    }


	@Override
	public Response search(@QueryParam("typeName") String typeName, @QueryParam("before") Date date) {
		logger.info("Searching for all [{}] with last modified before [{}]", typeName, date);
		SearchParameters searchParam = createSearchParam(typeName, 0, -1, Lists.<PropertyValue>newArrayList());
		searchParam.setBefore(date.getCalendar());
		List<String> ids = repositoryService.list(searchParam);
		logger.debug("Found ids: {}", ids);
		return ResponseFactory.ok(new RepositoryObjectIds(ids)).build();
	}
	
	private static final Logger logger = LoggerFactory.getLogger(QueryResource.class);
}
