package com.xebialabs.deployit.core.rest.api;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Lists;
import com.xebialabs.deployit.core.api.ControlTaskProxy;
import com.xebialabs.deployit.core.api.resteasy.http.tunnel.ResponseFactory;
import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.exception.DeployitException;
import com.xebialabs.deployit.exception.HttpResponseCodeResult;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.security.permission.Permission;
import com.xebialabs.deployit.service.controltask.ControlTaskService;
import com.xebialabs.deployit.task.Task;
import com.xebialabs.deployit.task.TaskStep;
import com.xebialabs.deployit.task.controltask.ControlTask;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.ws.rs.PathParam;
import javax.ws.rs.core.Response;

@Component
public class ControlTaskResource extends AbstractSecuredResource implements ControlTaskProxy {

	@Autowired
	private RepositoryService repository;
	
	@Autowired
	private ControlTaskService controlTaskService;

	@Override
	public Response execute(@PathParam("task") String controlTask, @PathParam("id") String configurationItemId) {
		checkPermission(Permission.CONTROLTASK_EXECUTE);
		ConfigurationItem read = repository.read(configurationItemId);
		ControlTask task = controlTaskService.execute(read, controlTask);
		if (task.getState() == Task.State.STOPPED || task.getState()  == Task.State.CANCELLED) {
			String logs = Joiner.on("\n").join(Lists.transform(task.getSteps(), new Function<TaskStep, String>() {
				public String apply(TaskStep input) {
					return input.getLog();
				}
			}));
			task.cancel();
			throw new TaskFailedException(controlTask, configurationItemId, logs);
		}
		return ResponseFactory.ok().build();
	}

	@SuppressWarnings("serial")
    @HttpResponseCodeResult(statusCode = 400)
	private class TaskFailedException extends DeployitException {
		public TaskFailedException(String controlTask, String configurationItemId, String logs) {
			super("ControlTask %s on configuration item %s failed.\n\nControl task log:\n\n%s", controlTask, configurationItemId, logs);
		}
	}
}
