package com.xebialabs.deployit.core.rest.api;

import com.google.common.base.Function;
import com.google.common.collect.Collections2;
import com.google.common.collect.ListMultimap;
import com.xebialabs.deployit.core.api.dto.ArtifactDto;
import com.xebialabs.deployit.core.api.dto.ConfigurationItemDto;
import com.xebialabs.deployit.core.api.dto.ConfigurationItemDtos;
import com.xebialabs.deployit.core.api.dto.Message;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage;
import com.xebialabs.deployit.util.PasswordObfuscator;
import com.xebialabs.overthere.OverthereFile;
import org.springframework.stereotype.Component;

import java.util.*;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.checks.Checks.checkNotNull;

@Component
public class ConfigurationItemDtoWriter {
	public ConfigurationItemDtos write(Collection<ConfigurationItem> cis) {
		ConfigurationItemDtos dtos = new ConfigurationItemDtos();
		for (ConfigurationItem ci : cis) {
			dtos.add(write(ci));
		}
		return dtos;
	}

	public ConfigurationItemDto write(ConfigurationItem ci) {
		ConfigurationItemDto dto;
		if (ci instanceof Artifact) {
			dto = new ArtifactDto(ci.getType().toString());
			dto.setId(ci.getId());
			OverthereFile file = ((Artifact) ci).getFile();
			if (file != null) {
				// Could be not set in case of Deployed generation
				((ArtifactDto) dto).setFilename(file.getName());
			}
		} else {
			dto = new ConfigurationItemDto(ci.getId(), ci.getType().toString());
		}
		copyValues(ci, dto);
		return dto;
	}

	@SuppressWarnings("unchecked")
    private void copyValues(ConfigurationItem ci, ConfigurationItemDto dto) {
		Descriptor descriptor = DescriptorRegistry.getDescriptor(ci.getType());
		for (PropertyDescriptor pd : descriptor.getPropertyDescriptors()) {
			Object value = pd.get(ci);
			if (value == null) continue;

			switch (pd.getKind()) {
				case STRING:
					if (pd.isPassword()) {
						value = PasswordObfuscator.ensureEncrypted((String) value);
					}
				case BOOLEAN:
				case INTEGER:
				case ENUM:
					dto.setProperty(pd.getName(), value.toString());
					break;
				case SET_OF_STRING:
					dto.setProperty(pd.getName(), newHashSet((Collection<String>) value));
					break;
				case SET_OF_CI:
					dto.setProperty(pd.getName(), handleSetOfCi((Collection<ConfigurationItem>) value));
					break;
				case CI:
					dto.setProperty(pd.getName(), ((ConfigurationItem) value).getId());
					break;
				case MAP_STRING_STRING:
					dto.setProperty(pd.getName(), newHashMap((Map<String, String>) value));
					break;
			}
		}
	}

	private Set<String> handleSetOfCi(Collection<ConfigurationItem> value) {
		return newHashSet(Collections2.transform(value, new Function<ConfigurationItem, String>() {
			public String apply(ConfigurationItem input) {
				return input.getId();
			}
		}));
	}

	public ConfigurationItemDto write(ConfigurationItem ci, List<ValidationMessage> messages) {
		ConfigurationItemDto dto = write(ci);
		for (ValidationMessage vm : messages) {
			dto.addValidation(new Message(vm.getPropertyName(), vm.getMessage()));
		}
		return dto;
	}

	public List<ConfigurationItemDto> write(final ListMultimap<Boolean, ConfigurationItem> deployeds) {
	    checkNotNull(deployeds, "deployeds");

		List<ConfigurationItemDto> cis = newArrayList();

	    for (ConfigurationItem validDeployedEntity : deployeds.get(true)) {
	        ConfigurationItemDto validDeployed = write(validDeployedEntity);
	        cis.add(validDeployed);
	    }

	    for (ConfigurationItem invalidDeployedEntity : deployeds.get(false)) {
	        ConfigurationItemDto invalidDeployed = write(invalidDeployedEntity);
	        ArrayList<Message> validations = new ArrayList<Message>();
	        validations.add(new Message("source", "The deployable for this deployed is missing from the package."));
	        invalidDeployed.setValidations(validations);
	        cis.add(invalidDeployed);
	    }

	    return cis;
	}

}
