package com.xebialabs.deployit.core.rest.api;


import com.google.common.base.Function;
import com.google.common.collect.Collections2;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.Lists;
import com.xebialabs.deployit.core.api.DeploymentProxy;
import com.xebialabs.deployit.core.api.dto.ConfigurationItemDto;
import com.xebialabs.deployit.core.api.dto.Deployment;
import com.xebialabs.deployit.core.api.dto.Steps;
import com.xebialabs.deployit.core.api.resteasy.http.tunnel.ResponseFactory;
import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.*;
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.repository.WorkDir;
import com.xebialabs.deployit.repository.WorkDirFactory;
import com.xebialabs.deployit.security.permission.Permission;
import com.xebialabs.deployit.server.api.util.IdGenerator;
import com.xebialabs.deployit.service.deployment.DeployedService;
import com.xebialabs.deployit.service.deployment.DeploymentService;
import com.xebialabs.deployit.service.validation.Validator;
import com.xebialabs.deployit.task.ExecutionEngine;
import com.xebialabs.deployit.task.Task;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Response;
import java.util.Collection;
import java.util.List;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.checks.Checks.checkNotNull;

@SuppressWarnings("deprecation")
@Controller
public class DeploymentResource extends AbstractSecuredResource implements DeploymentProxy {

	@Autowired
	private RepositoryService repositoryService;

	@Autowired
	private DeploymentService deploymentService;

	@Autowired
	private DeployedService deployedService;

	@Autowired
	private ConfigurationItemDtoWriter dtoWriter;

	@Autowired
	private ConfigurationItemDtoReader dtoReader;

	@Autowired
	private DtoWriter dtoConverter;

	@Autowired
	private ExecutionEngine engine;

	@Autowired
	private Validator validator;

	@Autowired
	private WorkDirFactory workDirFactory;

	@Override
	public Response prepareInitial(@QueryParam("version") String versionId, @QueryParam("environment") String environmentId) {
		logger.trace("prepareInitial {}, {}", versionId, environmentId);
		checkPermission(Permission.DEPLOY_INITIAL, newArrayList(environmentId));
		checkNotNull(versionId, "version");
		checkNotNull(environmentId, "environment");

		ConfigurationItem version = repositoryService.read(versionId);
		ConfigurationItem environment = repositoryService.read(environmentId);

		checkArgument(version instanceof Version, "%s is not a Version", versionId);
		checkArgument(environment instanceof Environment, "%s is not an Environment", environmentId);

		Deployment deployment = createDeployment((Version) version, (Environment) environment);

		return ResponseFactory.ok(deployment).build();
	}

	@Override
	public Response prepareUpgrade(@QueryParam("version") String newVersionId, @QueryParam("deployedApplication") String deployedApplicationId) {
		logger.trace("prepareUpgrade {}, {}", newVersionId, deployedApplicationId);
		checkNotNull(newVersionId, "version");
		checkNotNull(deployedApplicationId, "deployedApplication");
		WorkDir existingWorkDir = workDirFactory.newWorkDir();
		WorkDir newWorkDir = workDirFactory.newWorkDir();
		try {
			ConfigurationItem deployedApplication = repositoryService.read(deployedApplicationId, existingWorkDir);
			checkArgument(deployedApplication instanceof DeployedApplication, "%s is not a DeployedApplication", deployedApplicationId);

			Environment env = ((DeployedApplication) deployedApplication).getEnvironment();
			checkPermission(Permission.DEPLOY_UPGRADE, env.getId());

			ConfigurationItem newVersion = repositoryService.read(newVersionId, newWorkDir);
			checkArgument(newVersion instanceof Version, "%s is not a Version", newVersionId);

			ListMultimap<Boolean,ConfigurationItem> upgradedDeployeds = deployedService.generateUpgradedDeployeds((Version) newVersion, (DeployedApplication) deployedApplication);

			Deployment deployment = createDeployment((Version) newVersion, env);
			deployment.setUpgrade(true);
			deployment.setDeployeds(dtoWriter.write(upgradedDeployeds));

			return ResponseFactory.ok(deployment).build();
		} finally {
			existingWorkDir.delete();
			newWorkDir.delete();
		}
	}

	@Override
	public Response generateAllDeployeds(Deployment deployment) {
		logger.trace("generateAllDeployeds {}", deployment);
		checkNotNull(deployment, "deployment");
		checkPermission(Permission.DEPLOY_INITIAL, deployment.getEnvironment());

		WorkDir workDir = workDirFactory.newWorkDir();
		try {
			Version version = repositoryService.read(deployment.getVersion(), workDir);
			Environment environment = repositoryService.read(deployment.getEnvironment(), workDir);

			ListMultimap<Boolean, ConfigurationItem> initialDeployeds = deployedService.generateAllDeployeds(version, environment);

			deployment.addAll(dtoWriter.write(initialDeployeds));
			return ResponseFactory.ok(deployment).build();
		} finally {
			workDir.delete();
		}
	}

	@Override
	public Response generateSelectedDeployeds(@QueryParam("deployables") List<String> deployableIds, Deployment deployment) {
		logger.trace("generateSelectedDeployeds {}, {}", deployableIds, deployment);
		checkNotNull(deployment, "deployment");
		checkPermission(Permission.DEPLOY_INITIAL, deployment.getEnvironment());
		checkArgument(deployableIds.size() > 0, "Should select at least one deployable to generate a deployed");

		WorkDir workDir = workDirFactory.newWorkDir();
		try {
			Environment environment = repositoryService.read(deployment.getEnvironment());

			final List<ConfigurationItem> deployableCIs = Lists.newArrayList();
			for (String id : deployableIds) {
				deployableCIs.add(repositoryService.read(id));
				checkArgument(id.startsWith(deployment.getVersion()), "All sources should be from same package");
			}

			ListMultimap<Boolean, ConfigurationItem> selectedDeployeds = deployedService.generateSelectedDeployeds(deployableCIs, environment);
			deployment.addAll(dtoWriter.write(selectedDeployeds));
			return ResponseFactory.ok(deployment).build();
		} finally {
			workDir.delete();
		}
	}

	@Override
	public Response generateSingleDeployed(@QueryParam("deployable") String deployableId, @QueryParam("container") String containerId, @QueryParam("deployedtype") String deployedType, Deployment deployment) {
		checkNotNull(deployment, "deployment");
		checkPermission(Permission.DEPLOY_INITIAL, deployment.getEnvironment());

		checkNotNull(deployableId, "deployable");
		checkNotNull(containerId, "container");

		WorkDir workDir = workDirFactory.newWorkDir();
		try {
			ConfigurationItem deployable = repositoryService.read(deployableId);
			checkArgument(deployable instanceof Deployable, "%s should be a Deployable.", deployableId);
			ConfigurationItem container = repositoryService.read(containerId);
			checkArgument(container instanceof Container, "%s should be a Container.", containerId);
			Environment environment = repositoryService.read(deployment.getEnvironment());

			ListMultimap<Boolean, ConfigurationItem> deployeds = deployedService.generateSelectedDeployed((Deployable) deployable, (Container) container, deployedType, environment);
			deployment.addAll(dtoWriter.write(deployeds));
			return ResponseFactory.ok(deployment).build();
		} finally {
			workDir.delete();
		}
	}

	@Override
	public Response validate(Deployment deployment) {
		if (!validateDeployedsWithValidator(deployment.getDeployeds())) {
			return ResponseFactory.badRequest(deployment).build();
		}

		return ResponseFactory.ok(deployment).build();
	}

	@Override
	public Response deploy(Deployment deployment) {
		logger.trace("deploy {}", deployment);
		// TODO in case of an upgrade, we need to verify that a user on the CLI didn't break the security contract by manually adding extra deployeds...
		checkNotNull(deployment, "deployment");
		Task deploymentTask = createDeploymentTask(deployment);
		String taskId = engine.register(deploymentTask);
		Steps steps = dtoConverter.taskToDto(deploymentTask);
		steps.setTaskId(taskId);
		return ResponseFactory.ok(steps).build();
	}

	@Override
	public Response undeploy(String deployedApplicationId) {
		WorkDir undeploymentWorkDir = workDirFactory.newWorkDir();
		try {
			checkNotNull(deployedApplicationId, "deployedApplication");
			ConfigurationItem deployedApplication = repositoryService.read(deployedApplicationId, undeploymentWorkDir);
			checkArgument(deployedApplication instanceof DeployedApplication, "%s is not a DeployedApplication", deployedApplicationId);
			checkPermission(Permission.UNDEPLOY, ((DeployedApplication) deployedApplication).getEnvironment().getId());
			Task task = deploymentService.prepareUndeployment((DeployedApplication) deployedApplication, undeploymentWorkDir);
			String taskId = engine.register(task);
			Steps steps = dtoConverter.taskToDto(task);
			steps.setTaskId(taskId);
			return ResponseFactory.ok(steps).build();
		} catch(RuntimeException e) {
			undeploymentWorkDir.delete();
			throw e;			
		}
	}

	private Task createDeploymentTask(Deployment deployment) {
		WorkDir workDir = workDirFactory.newWorkDir();
		Task deploymentTask = null;
		try {
			Version version = repositoryService.read(deployment.getVersion(), workDir);

			if (!deployment.isUpgrade()) {
				Collection<ConfigurationItem> deployedEntities = dtoReader.read(deployment.getDeployeds(), workDir);
				Environment environment = repositoryService.read(deployment.getEnvironment(), workDir);
				deploymentTask = deploymentService.prepareInitialDeployment(version, environment, toDeployeds(deployedEntities), workDir);
			} else {
				WorkDir existingDeploymentWorkDir = workDirFactory.newWorkDir();
				try {
					Collection<ConfigurationItem> deployedEntities = dtoReader.read(deployment.getDeployeds(), workDir);
					DeployedApplication existingDeployedApplication = repositoryService.read(deployment.getDeployedApplication().getId(), existingDeploymentWorkDir);
					deploymentTask = deploymentService.prepareUpgradeDeployment(version, existingDeployedApplication, toDeployeds(deployedEntities), existingDeploymentWorkDir, workDir);
				} catch (RuntimeException e) {
					existingDeploymentWorkDir.delete();
					throw e;
				}
			}
		} catch (RuntimeException e) {
			// Only delete the workdir in case an exception is thrown, as the task needs the files...
			workDir.delete();
			throw e;
		}
		return deploymentTask;
	}

	@SuppressWarnings("rawtypes")
    private Collection<Deployed> toDeployeds(Collection<ConfigurationItem> deployedEntities) {
		return Collections2.transform(deployedEntities, new Function<ConfigurationItem, Deployed>() {
			@Override
			public Deployed apply(ConfigurationItem input) {
				checkArgument(input instanceof Deployed, "%s should be a Deployed", input.getId());
				return (Deployed) input;
			}
		});
	}


	private boolean validateDeployedsWithValidator(final List<ConfigurationItemDto> deployeds) {
		boolean allValid = true;
        List<ConfigurationItem> deployedEntitiesInCtx = dtoReader.read(deployeds);

        for (int i=0;i<deployeds.size();i++) {
            final ConfigurationItemDto deployed = deployeds.get(i);
            final ConfigurationItem deployedEntity = deployedEntitiesInCtx.get(i);
	        List<ValidationMessage> messages = validator.validate(deployedEntity, deployedEntitiesInCtx);
	        if (!messages.isEmpty()) {
				deployed.setValidations(dtoWriter.write(deployedEntity, messages).getValidations());
				allValid = false;
			}
		}
		return allValid;
	}

	private Deployment createDeployment(Version version, Environment env) {
		Deployment deployment = new Deployment();

		DeployedApplication deployedApplication = DescriptorRegistry.getDescriptor(Type.valueOf(DeployedApplication.class)).newInstance();
		deployedApplication.setVersion(version);
		deployedApplication.setEnvironment(env);

		IdGenerator.generateId(deployedApplication);

		deployment.setDeployedApplication(dtoWriter.write(deployedApplication));
		return deployment;
	}

	private static final Logger logger = LoggerFactory.getLogger(DeploymentResource.class);
}
