package com.xebialabs.deployit.core.rest.api.reports;

import static com.xebialabs.deployit.task.Task.DEPLOYMENT_PREFIX;
import static com.xebialabs.deployit.task.Task.UPGRADE_PREFIX;
import static com.xebialabs.deployit.task.Task.State.DONE;
import static com.xebialabs.deployit.task.Task.State.STOPPED;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.commons.lang.StringUtils;

import com.google.common.base.Predicate;
import com.google.common.collect.Collections2;
import com.google.common.collect.Maps;
import com.xebialabs.deployit.core.api.dto.Report;
import com.xebialabs.deployit.core.api.resteasy.Date;
import com.xebialabs.deployit.core.rest.api.ArchivedTaskSearchParameterFactory;
import com.xebialabs.deployit.repository.ArchivedTaskSearchParameters;
import com.xebialabs.deployit.task.Task;
import com.xebialabs.deployit.task.TaskArchive;
import com.xebialabs.deployit.task.labelparser.DeploymentTaskLabelParser;
import com.xebialabs.deployit.task.labelparser.InitialDeploymentTaskLabelParser;
import com.xebialabs.deployit.task.labelparser.UpgradeTaskLabelParser;

public class DeploymentsStateBreakdownReport {

	private TaskArchive taskArchive;

	public DeploymentsStateBreakdownReport(TaskArchive taskArchive) {
		this.taskArchive = taskArchive;
	}

	public Report report(final List<String> repositoryEntityIds, final Date startDate, final Date endDate,
			final String filterType) {
		Report report = new Report();
		Map<String, TaskStateData> tasksMap = Maps.newHashMap();
		
		ArchivedTaskSearchParameters archivedTaskSearchParameters = ArchivedTaskSearchParameterFactory.createArchivedTaskSearchParameters(startDate, endDate);

		final Collection<Task> filteredTasks = Collections2.filter(taskArchive.searchTasks(archivedTaskSearchParameters)
				, new Predicate<Task>() {
			@Override
			public boolean apply(final Task input) {
				String entityId = "";
				entityId = getEntityIdFromDeploymentTask(input, filterType);
				if (StringUtils.isEmpty(entityId) || !repositoryEntityIds.contains(entityId)) {
					return false;
				}
				return true;
			}
		});

		for (Task task : filteredTasks) {
			createFilteredTasksMap(tasksMap, task, filterType);
		}

		for (Iterator<String> iterator = repositoryEntityIds.iterator(); iterator.hasNext();) {
			String entityId = (String) iterator.next();
			entityId = entityId.substring(entityId.indexOf("/") + 1);
			if (!tasksMap.containsKey(entityId)) {
				tasksMap.put(entityId, new TaskStateData(new AtomicInteger(0), new AtomicInteger(0)));
			}
		}
		report = generateReportDataFromTaskMap(tasksMap);
		return report;
	}

	private Report generateReportDataFromTaskMap(Map<String, TaskStateData> tasksMap) {
		final Report report = new Report();
		for (Map.Entry<String, TaskStateData> entry : tasksMap.entrySet()) {
			final Report.ReportLine reportLine = report.addLine();
			reportLine.addValue("entityName", entry.getKey());
			reportLine
					.addValue("noOfSuccessfulDeployments", entry.getValue().getNoOfSuccessfulDeployments().toString());
			reportLine.addValue("noOfFailedDeployments", entry.getValue().getNoOfFailedDeployments().toString());
		}
		return report;
	}

	private void createFilteredTasksMap(Map<String, TaskStateData> tasksMap, Task task, String filterType) {
		String entityId = getEntityIdFromDeploymentTask(task, filterType);
		entityId = entityId.substring(entityId.indexOf("/") + 1);
		if (task.getState().equals(DONE)) {
			if (!tasksMap.containsKey(entityId)) {
				tasksMap.put(entityId, new TaskStateData(new AtomicInteger(1), new AtomicInteger(0)));
			} else {
				tasksMap.get(entityId).getNoOfSuccessfulDeployments().incrementAndGet();
			}
		} else if (task.getState().equals(STOPPED)) {
			if (!tasksMap.containsKey(entityId)) {
				tasksMap.put(entityId, new TaskStateData(new AtomicInteger(0), new AtomicInteger(1)));
			} else {
				tasksMap.get(entityId).getNoOfFailedDeployments().incrementAndGet();
			}
		}
	}

	private String getEntityIdFromDeploymentTask(Task inputTask, String filterType) {
		String entityId = "";
		DeploymentTaskLabelParser labelParser = null;
		String taskLabel = inputTask.getLabel();
		if (taskLabel.startsWith(DEPLOYMENT_PREFIX)) {
			labelParser = new InitialDeploymentTaskLabelParser(taskLabel);
		} else if (taskLabel.startsWith(UPGRADE_PREFIX)) {
			labelParser = new UpgradeTaskLabelParser(taskLabel);
		}
		if (labelParser != null) {
			if (filterType.equals(ReportFilterType.ENVIRONMENT.getFilterType())) {
				entityId = labelParser.getEnvironment();
			}else if (filterType.equals(ReportFilterType.APPLICATION.getFilterType())) {
				entityId = labelParser.getApplication();
			}
		}
		return entityId;
	}

}
