package com.xebialabs.deployit.core.rest.api.reports;

import static com.xebialabs.deployit.core.api.resteasy.Date.FORMAT;
import static com.xebialabs.deployit.task.Task.DEPLOYMENT_PREFIX;
import static com.xebialabs.deployit.task.Task.UNDEPLOYMENT_PREFIX;
import static com.xebialabs.deployit.task.Task.UPGRADE_PREFIX;
import static com.xebialabs.deployit.task.Task.State.CANCELLED;
import static com.xebialabs.deployit.task.Task.State.DONE;

import java.util.Calendar;
import java.util.Collection;

import com.google.common.base.Predicate;
import com.google.common.collect.Collections2;
import com.xebialabs.deployit.core.api.dto.Report;
import com.xebialabs.deployit.core.api.resteasy.Date;
import com.xebialabs.deployit.task.Task;
import com.xebialabs.deployit.task.Task.State;
import com.xebialabs.deployit.task.TaskRegistry;
import com.xebialabs.deployit.task.labelparser.DeploymentTaskLabelParser;
import com.xebialabs.deployit.task.labelparser.InitialDeploymentTaskLabelParser;
import com.xebialabs.deployit.task.labelparser.UndeploymentTaskLabelParser;
import com.xebialabs.deployit.task.labelparser.UpgradeTaskLabelParser;

public class TasksInDateRangeReport {

	private TaskRegistry taskRegistry;

	public TasksInDateRangeReport(final TaskRegistry taskRegistry) {
		this.taskRegistry = taskRegistry;
	}

	public Report report(final Date begin, final Date end) {
		final Collection<Task> tasksInRange = Collections2.filter(taskRegistry.getAllArchivedTasks(), new Predicate<Task>() {
			@Override
			public boolean apply(final Task input) {
				final Calendar completionDate = input.getCompletionDate();
				State state = input.getState();
				boolean isValidState = state.equals(DONE) || state.equals(CANCELLED);
				if (!isValidState || !input.isDeploymentTask() || completionDate == null) {
					return false;
				}
				boolean isAfterBegin = completionDate.after(begin.getCalendar());
				boolean isEqualsBegin = completionDate.equals(begin.getCalendar());
				boolean isBeforeEnd = completionDate.before(end.getCalendar());
				return (isAfterBegin || isEqualsBegin) && isBeforeEnd;
			}
		});

		final Report report = new Report();

		for (Task task : tasksInRange) {
			final Report.ReportLine reportLine = report.addLine();
			reportLine.addValue("user", task.getOwner());
			reportLine.addValue("status", task.getState().name());
			reportLine.addValue("startDate", new Date(task.getStartDate(), FORMAT).toString());
			reportLine.addValue("completionDate", new Date(task.getCompletionDate(), FORMAT).toString());
			fillParsedInfo(reportLine, task.getLabel());
		}

		return report;
	}

	private void fillParsedInfo(Report.ReportLine reportLine, String taskLabel) {
		String taskType = "Unknown";
		DeploymentTaskLabelParser labelParser = null;
		if (taskLabel.startsWith(DEPLOYMENT_PREFIX)) {
			taskType = "Initial";
			labelParser = new InitialDeploymentTaskLabelParser(taskLabel);
		} else if (taskLabel.startsWith(UPGRADE_PREFIX)) {
			taskType = "Upgrade";
			labelParser = new UpgradeTaskLabelParser(taskLabel);
		} else if (taskLabel.startsWith(UNDEPLOYMENT_PREFIX)) {
			taskType = "Undeployment";
			labelParser = new UndeploymentTaskLabelParser(taskLabel);
		}
		if (labelParser != null) {
			String packageId = labelParser.getApplication();
			String enviromentId = labelParser.getEnvironment();
			reportLine.addValue("type", taskType);
			reportLine.addValue("package", packageId.substring(packageId.indexOf("/") + 1) + "/" + labelParser.getVersion());
			reportLine.addValue("environment", enviromentId.substring(enviromentId.indexOf("/") + 1));
		}
	}
}
