package com.xebialabs.deployit.core.rest.api;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.checks.Checks.checkNotNull;
import static org.apache.commons.lang.StringUtils.isNotBlank;

import java.io.ByteArrayInputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.xebialabs.deployit.Mappings;
import com.xebialabs.deployit.checks.Checks.IncorrectArgumentException;
import com.xebialabs.deployit.core.api.dto.Archetype;
import com.xebialabs.deployit.core.api.dto.Artifact;
import com.xebialabs.deployit.core.api.dto.ArtifactAndData;
import com.xebialabs.deployit.core.api.dto.ConfigurationItem;
import com.xebialabs.deployit.core.api.dto.ConfigurationItemDescriptorDto;
import com.xebialabs.deployit.core.api.dto.ConfigurationItemPropertyDescriptorDto;
import com.xebialabs.deployit.core.api.dto.ConfigurationItemPropertyType;
import com.xebialabs.deployit.core.api.dto.Message;
import com.xebialabs.deployit.core.api.dto.RepositoryObject;
import com.xebialabs.deployit.core.api.dto.RepositoryObjects;
import com.xebialabs.deployit.core.api.dto.StepInfo;
import com.xebialabs.deployit.core.api.dto.Steps;
import com.xebialabs.deployit.core.api.dto.TaskInfo;
import com.xebialabs.deployit.core.api.dto.TaskInfos;
import com.xebialabs.deployit.plugin.PojoConverter;
import com.xebialabs.deployit.reflect.ConfigurationItemDescriptor;
import com.xebialabs.deployit.reflect.ConfigurationItemPropertyDescriptor;
import com.xebialabs.deployit.repository.ArchetypeEntity;
import com.xebialabs.deployit.repository.ArtifactEntity;
import com.xebialabs.deployit.repository.ConfigurationItemEntity;
import com.xebialabs.deployit.repository.RepositoryObjectEntity;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.service.validation.ValidationMessage;
import com.xebialabs.deployit.task.Task;
import com.xebialabs.deployit.task.TaskStep;
import com.xebialabs.deployit.typedescriptor.ConfigurationItemTypeDescriptorRepository;

/**
 */
@Component
public class DtoConverter {

	private RepositoryService repositoryService;

	private PojoConverter pojoConverter;

	private ConfigurationItemTypeDescriptorRepository descriptorRepository;
	
	@Autowired
	public DtoConverter(RepositoryService repositoryService, PojoConverter pojoConverter, ConfigurationItemTypeDescriptorRepository descriptorRepository) {
		this.repositoryService = repositoryService;
		this.pojoConverter = pojoConverter;
		this.descriptorRepository = descriptorRepository;
	}

    public RepositoryObject toDto(final RepositoryObjectEntity entity) {
        checkNotNull(entity, "entity");
        checkNotNull(entity.getId(), "entity.id");
        checkNotNull(entity.getConfigurationItemTypeName(), "entity.type");

        RepositoryObject dto;
        if (entity instanceof ArtifactEntity) {
            dto = new Artifact();
        } else if (entity instanceof ConfigurationItemEntity) {
            dto = new ConfigurationItem();
        } else if (entity instanceof ArchetypeEntity) {
            dto = new Archetype();
        } else {
            throw new IllegalArgumentException("entity is not of class " + ConfigurationItemEntity.class.getName() + " or class "
                + ArchetypeEntity.class.getName());
        }

        ArchetypeEntity archetype = entity.getConfigurationItemArchetype();
        if (archetype != null) {
            dto.setArchetypeId(archetype.getId());
        }

        dto.setId(entity.getId());
        dto.setConfigurationItemTypeName(entity.getConfigurationItemTypeName());
        dto.setLastModified(entity.getLastModified());
        dto.setOverrideLastModified(null); // Default value for DTO
        dto.setCreatingTaskId(entity.getCreatingTaskId());
        copyValuesToDto(entity, dto);
        return dto;
    }

    private void copyValuesToDto(final RepositoryObjectEntity entity, final RepositoryObject dto) {
        final HashMap<String, Object> map = Maps.newHashMap();
        final ConfigurationItemDescriptor descriptor = descriptorRepository.getDescriptorByType(entity.getConfigurationItemTypeName());
        for (Map.Entry<String, Object> entry : entity.getValues().entrySet()) {
            String key = entry.getKey();
            Object value = entry.getValue();
            final ConfigurationItemPropertyDescriptor propertyDescriptor = descriptor.getPropertyDescriptor(entry.getKey());
            if (propertyDescriptor != null && propertyDescriptor.isPassword()) {
                value = "********";
            }

            map.put(key, value);
        }

        dto.setValues(map);
    }

    public RepositoryObjectEntity fromDto(final RepositoryObject dto) {
        checkNotNull(dto, "dto");
        checkNotNull(dto.getId(), "dto.id");

        RepositoryObjectEntity entity = createEntity(dto);

	    entity.setId(dto.getId());
        entity.setLastModified(null); // Cannot be overridden from DTO
        entity.setOverrideLastModified(dto.getOverrideLastModified());
        entity.setCreatingTaskId(null); // Cannot be overridden from DTO
	    copyValues(dto, entity);
        if (dto instanceof Artifact) {
            entity.addValue("location", "jcr://" + entity.getId()); // Cannot be overridden from DTO
        }
        return entity;
    }

	private void copyValues(RepositoryObject from, RepositoryObjectEntity to) {
		for (String fromKey : from.getValues().keySet()) {
			to.addValue(fromKey, from.getValues().get(fromKey));
		}
	}

	private RepositoryObjectEntity createEntity(RepositoryObject dto) {
		RepositoryObjectEntity entity;
		if (dto instanceof Artifact) {
		    entity = createObjectWithDefaults(dto.getConfigurationItemTypeName());
		} else if (dto instanceof ConfigurationItem) {
		    if (isNotBlank(dto.getArchetypeId())) {
		        ArchetypeEntity archetype = repositoryService.read(dto.getArchetypeId());
			    if (isNotBlank(dto.getConfigurationItemTypeName())) {
			        entity = new ConfigurationItemEntity(dto.getConfigurationItemTypeName());
				    entity.setArchetype(archetype);
			    } else {
				    entity = new ConfigurationItemEntity(archetype);
			    }
		    } else {
		        entity = createObjectWithDefaults(dto.getConfigurationItemTypeName());
		    }
		} else {
		    if (isNotBlank(dto.getArchetypeId())) {
		        ArchetypeEntity archetype = repositoryService.read(dto.getArchetypeId());
			    if (isNotBlank(dto.getConfigurationItemTypeName())) {
					entity = new ArchetypeEntity(dto.getConfigurationItemTypeName());
					entity.setArchetype(archetype);
			    } else {
				    entity = new ArchetypeEntity(archetype);
			    }
		    } else {
			    final ConfigurationItemEntity defaults = createObjectWithDefaults(dto.getConfigurationItemTypeName());
			    entity = new ArchetypeEntity(dto.getConfigurationItemTypeName());
			    entity.setValues(defaults.getValues());
		    }
		}
		return entity;
	}

	private ConfigurationItemEntity createObjectWithDefaults(String ciType) {
		checkNotNull(ciType, "dto.type");
		final ConfigurationItemDescriptor descriptor = descriptorRepository.getDescriptorByType(ciType);
		if(descriptor == null) {
			throw new IncorrectArgumentException("Configuration item type " + ciType + " is not known");
		}
		final Object ciPojo = descriptor.newInstance();
		return pojoConverter.toEntity(ciPojo);
	}

	public ArtifactEntity fromDto(final ArtifactAndData dto) {
        checkNotNull(dto, "dto");
        checkNotNull(dto.getArtifact(), "dto.artifact");
        checkNotNull(dto.getArtifact().getId(), "dto.artifact.id");
        checkNotNull(dto.getData(), "dto.artifact.data");

        ArtifactEntity entity = (ArtifactEntity) fromDto(dto.getArtifact());
        entity.setData(new ByteArrayInputStream(dto.getData()));
        entity.addValue("location", "jcr://" + entity.getId());
        return entity;
    }

    public RepositoryObjects toDto(final Collection<? extends RepositoryObjectEntity> entities) {
        checkNotNull(entities, "entities");

        final RepositoryObjects objects = new RepositoryObjects();
        for (RepositoryObjectEntity each : entities) {
            objects.add(toDto(each));
        }
        return objects;
    }

    public RepositoryObjects toDto(final Mappings mappings) {
    	checkNotNull(mappings, "mappings");
    	
    	final RepositoryObjects objects = new RepositoryObjects();
    	for (RepositoryObjectEntity validMappingEntity : mappings.getValidMappings()) {
    		RepositoryObject validMapping = toDto(validMappingEntity);
    		objects.add(validMapping);
    	}
    	for (RepositoryObjectEntity invalidMappingEntity : mappings.getInvalidMappings()) {
    		RepositoryObject invalidMapping = toDto(invalidMappingEntity);
    		ArrayList<Message> validations = new ArrayList<Message>();
    		validations.add(new Message("source", "A valid source for this mapping could not be found to upgrade."));
			invalidMapping.setValidations(validations );
    		objects.add(invalidMapping);
    	}
    	return objects;
    }

    @SuppressWarnings("unchecked")
    public <T extends RepositoryObjectEntity> Collection<T> fromDto(final RepositoryObjects objects) {
        checkNotNull(objects, "objects");

        List<T> entities = newArrayList();
        for (RepositoryObject each : objects.getObjects()) {
            entities.add((T) fromDto(each));
        }
        return entities;
    }

    public ConfigurationItemDescriptorDto toDto(final ConfigurationItemDescriptor descriptor) {
        checkNotNull(descriptor, "descriptor");

        final ConfigurationItemDescriptorDto dto = new ConfigurationItemDescriptorDto();
        dto.setType(descriptor.getType());
        dto.setSimpleName(descriptor.getSimpleName());
        dto.setRootName(descriptor.getRoot().getRootNodeName() != null ? descriptor.getRoot().getRootNodeName() : "");
        dto.setInterfaces(descriptor.getInterfaces());
        dto.setSuperClasses(descriptor.getSuperClasses());
        dto.setMappingSourceType(descriptor.getMappingSourceType());
        dto.setMappingTargetType(descriptor.getMappingTargetType());
        dto.setDescription(descriptor.getDescription());
        dto.setPlaceholderFormatName(descriptor.getPlaceholderFormatName());
        dto.setPlaceholdersName(descriptor.getPlaceholdersName());

        ConfigurationItemPropertyDescriptor[] originalPropertyDescriptors = descriptor.getPropertyDescriptors();
        for (ConfigurationItemPropertyDescriptor propertyDescriptor : originalPropertyDescriptors) {
            dto.addPropertyDescriptor(toDto(propertyDescriptor));
        }

        return dto;
    }

    public ConfigurationItemPropertyDescriptorDto toDto(final ConfigurationItemPropertyDescriptor propertyDescriptor) {
        checkNotNull(propertyDescriptor, "propertyDescriptor");

        final ConfigurationItemPropertyDescriptorDto dto = new ConfigurationItemPropertyDescriptorDto();
        dto.setName(propertyDescriptor.getName());
        dto.setLabel(propertyDescriptor.getLabel());
        dto.setType(ConfigurationItemPropertyType.valueOf(propertyDescriptor.getType().toString()));
        dto.setEnumValues(propertyDescriptor.getEnumValues());
        dto.setCollectionMemberClassname(propertyDescriptor.getCollectionMemberClassname());
        dto.setPropertyClassname(propertyDescriptor.getPropertyClassname());
        dto.setRequired(propertyDescriptor.isRequired());
        dto.setEditable(propertyDescriptor.isEditable());
        dto.setPassword(propertyDescriptor.isPassword());
        dto.setIdentifying(propertyDescriptor.isIdentifying());
        dto.setDiscoveryParam(propertyDescriptor.isDiscoveryParam());
        dto.setDiscoveryRequired(propertyDescriptor.isDiscoveryRequired());
        dto.setCategory(propertyDescriptor.getCategory());
        dto.setDefaultValue(propertyDescriptor.getDefaultValue());
        dto.setDescription(propertyDescriptor.getDescription());
        dto.setAsContainment(propertyDescriptor.asContainment());
        ConfigurationItemPropertyDescriptor[] originalPropertyDescriptors = propertyDescriptor.getListObjectPropertyDescriptors();
        if (originalPropertyDescriptors != null) {
            for (ConfigurationItemPropertyDescriptor listObjectPropertyDescriptor : originalPropertyDescriptors) {
                dto.addPropertyDescriptor(toDto(listObjectPropertyDescriptor));
            }
        }
        return dto;
    }

    public TaskInfo toDto(String id, Task task) {
        checkNotNull(id, "id");
        checkNotNull(task, "task");

        TaskInfo ti = new TaskInfo();
        ti.setId(id);
        ti.setLabel(task.getLabel());
        ti.setState(task.getState().name());
        ti.setStartDate(task.getStartDate());
        ti.setCompletionDate(task.getCompletionDate());
        ti.setNrOfSteps(task.getNrOfSteps());
        ti.setCurrentStepNr(task.getCurrentStepNr());
        if (task.getOwner() != null) {
            ti.setUser(task.getOwner());
        }
        return ti;
    }

    public StepInfo toDto(int stepNr, TaskStep step) {
        checkNotNull(step, "step");

        StepInfo si = new StepInfo();
        si.setNr(stepNr);
        si.setDescription(step.getDescription());
        si.setState(step.getState().name());
        si.setStartDate(step.getStartDate());
        si.setCompletionDate(step.getCompletionDate());
        si.setLog(step.getLog());

        return si;
    }

    public Steps toDto(Task deploymentTask) {
        Steps steps = new Steps();
        List<TaskStep> stepList = deploymentTask.getSteps();
        int stepNr = 1;
        for (TaskStep taskStep : stepList) {
            steps.add(toDto(stepNr++, taskStep));
        }
        return steps;
    }

	public TaskInfos toTaskInfos(List<Task> tasks) {
		TaskInfos taskInfos = new TaskInfos();
		for (Task eachTask : tasks) {
			taskInfos.add(toDto(eachTask.getId(), eachTask));
		}
		return taskInfos;
	}

    public RepositoryObject toDto(final RepositoryObjectEntity entity, final List<ValidationMessage> messages) {
        final RepositoryObject repositoryObject = toDto(entity);
        repositoryObject.setValidations(toDto(messages));
        return repositoryObject;
    }

    public List<Message> toDto(final List<ValidationMessage> messages) {
        final List<Message> list = Lists.newArrayList();
        for (ValidationMessage validationMessage : messages) {
            list.add(new Message(validationMessage.getFieldName(), validationMessage.getMessage()));
        }

        return list;
    }

}
