package com.xebialabs.deployit.core.util

import ai.digital.deploy.settings.GeneralSettings
import com.xebialabs.deployit.checks.Checks
import org.apache.tika.Tika

import java.util.Base64

object GeneralSettingsUtils {

  // 1 MB
  private val MAX_LOGO_FILE_SIZE_BYTES = 1048576
  private val MAX_INSTANCE_NAME_SIZE = 150

  private val tika = new Tika()
  private val base64 = "base64,"

  private object Duration extends Enumeration {
    private type Duration = Value
    val SEVEN: Value = Value(7)
    val THIRTY: Value = Value(30)
    val FORTY_FIVE: Value = Value(45)
    val SIXTY: Value = Value(60)
    val THREE_HUNDRED_SIXTY_FIVE: Value = Value(365)
    val NO_EXPIRATION: Value = Value(-1)

    def fromValue(value: Int): Option[Duration] = {
      values.find(_.id == value)
    }
  }

  def validate(generalSetting: GeneralSettings): Unit = {
    validateInstanceNameLength(generalSetting.instanceName)
    validateLogoFile(generalSetting.logoFile)
    validateDurationValue(generalSetting.patExpirationDuration)
    validatePatNotificationConfig(generalSetting.patEmailNotificationEnabled, generalSetting.patEmailNotificationTriggerPeriod, generalSetting.patEmailNotificationSMTPServerCiRef)
  }

  private def validateInstanceNameLength(instanceName: String): Unit = {
    if(instanceName != null) {
      Checks.checkTrue(instanceName.length <= MAX_INSTANCE_NAME_SIZE, s"Instance name length should be maximum $MAX_INSTANCE_NAME_SIZE characters.")
    }
  }

  private def validateLogoFile(logoFile: String): Unit = {
    if (logoFile != null && logoFile.nonEmpty) {
      validateLogoFileSize(logoFile)
      validateLogoFileType(logoFile)
      validateLogoMimeType(logoFile)
    }
  }

  private def validateLogoFileSize(logoFile: String): Unit = {
    val characterCount = logoFile.length;
    val paddingCount = logoFile.substring(characterCount - 2, characterCount).count(c => c == '=')
    val sizeInBytes = (3 * (characterCount / 4)) - paddingCount;
    Checks.checkTrue(sizeInBytes <= MAX_LOGO_FILE_SIZE_BYTES, "Logo file is too large. Maximum size is 1 MB")
  }

  private def validateLogoFileType(logoFile: String): Unit = {
    Checks.checkTrue(logoFile.substring("data:".length).startsWith("image/"), "Logo file must be of an image file type.")
  }

  private def validateLogoMimeType(logoFile: String): Unit = {
    val detectedMimeType = getLogoMimeType(logoFile)
    Checks.checkTrue(detectedMimeType.startsWith("image/"), "Logo file must be of an image file type.")
  }

  def getLogoMimeType(logoFile: String): String = {
    val decodedBytes = Base64.getDecoder.decode(logoFile.substring(logoFile.indexOf(base64)+base64.length))
    tika.detect(decodedBytes)
  }

  private def validateDurationValue(duration: String): Unit = {
    if (duration != null && duration.nonEmpty && (duration.forall(_.isDigit) || (duration.startsWith("-") && duration.substring(1).forall(_.isDigit)))) {
      Duration.fromValue(duration.toInt) match {
        case Some(_) =>
        case None => throw new IllegalArgumentException(s"Invalid duration value: $duration")
      }
    } else {
      throw new IllegalArgumentException("Duration value must be a valid integer and cannot be null or empty.")
    }
  }

  private def validatePatNotificationConfig(patEmailNotificationEnabled: Boolean, patEmailNotificationTriggerPeriod: String, patEmailNotificationSMTPServerCiRef: String): Unit = {
    if(patEmailNotificationEnabled) {
      if (patEmailNotificationTriggerPeriod == null || patEmailNotificationTriggerPeriod.isEmpty) {
        throw new IllegalArgumentException("PAT email notification trigger period must be specified when PAT notifications are enabled.")
      }
      if (patEmailNotificationSMTPServerCiRef == null || patEmailNotificationSMTPServerCiRef.isEmpty) {
        throw new IllegalArgumentException("PAT email notification SMTP server must be specified when PAT notifications are enabled.")
      }
    }
  }
}
