package com.xebialabs.deployit.core.rest.api.reports

import com.xebialabs.deployit.checks.Checks.checkArgument
import com.xebialabs.deployit.security.Permissions
import com.xebialabs.deployit.task.archive.{StatusOverviewDataReportLine, StatusOverviewReportLine}
import org.apache.poi.common.usermodel.HyperlinkType
import org.apache.poi.ss.usermodel.{Font, Workbook}
import org.apache.poi.xssf.usermodel.{XSSFFont, XSSFWorkbook}
import org.joda.time.{DateTime, LocalDateTime}
import org.joda.time.format.{DateTimeFormat, DateTimeFormatter}

import java.lang.String.format
import java.text.{DecimalFormat, DecimalFormatSymbols}
import java.util
import java.util.Locale
import scala.jdk.CollectionConverters.CollectionHasAsScala

object DeploymentReportsUtility {
  private val DEPLOYMENT_DATE_FORMAT_FILE_REPORT: DateTimeFormatter = DateTimeFormat.forPattern("MMMM d, yyyy HH:mm a")

  def createDeploymentsDashboardReport(begin: DateTime,
                             end: DateTime,
                             dashboardReport: util.Collection[StatusOverviewReportLine],
                             deploymentReport: util.Collection[StatusOverviewDataReportLine],
                             serverName: String,
                             versionNumber: String): Workbook = {
    val workbook = new XSSFWorkbook
    createTemplateTab(workbook, serverName, versionNumber, begin, end)
    createDashboardReportTab(workbook, dashboardReport)
    createDeploymentReportTab(workbook, deploymentReport)
    workbook
  }

  private def createTemplateTab(workbook: XSSFWorkbook, serverName: String, versionNumber: String, begin: DateTime,
                                end: DateTime): Unit = {
    val userName = Permissions.getAuthenticatedUserName
    val dateTime = format("%s", DEPLOYMENT_DATE_FORMAT_FILE_REPORT.print(LocalDateTime.now))
    val period = format("%s", DEPLOYMENT_DATE_FORMAT_FILE_REPORT.print(begin) + " to " +
      DEPLOYMENT_DATE_FORMAT_FILE_REPORT.print(end))
    val version = format("%s", "Deploy " + versionNumber)
    val sheet = workbook.createSheet("Report Info")
    sheet.setDefaultRowHeightInPoints(19)
    val reportInfo = Array("Instance URL", "Report Generated On", "Generated By", "Version", "Period")
    val reportInfoValue = Array(serverName, dateTime, userName, version, period)

    for (i <- 0 to 4) {
      val row = sheet.createRow(i)
      row.createCell(0).setCellValue(reportInfo(i))
      row.createCell(1).setCellValue(reportInfoValue(i))

      if (i == 0) {
        val createHelper = workbook.getCreationHelper
        val link = createHelper.createHyperlink(HyperlinkType.URL)
        val hLinkFont: XSSFFont = workbook.createFont
        val hLinkStyle = workbook.createCellStyle
        hLinkFont.setUnderline(Font.U_SINGLE)
        hLinkStyle.setFont(hLinkFont)
        link.setAddress(serverName)
        val cell = row.getCell(1)
        cell.setCellStyle(ExcelStyle.getHyperlinkCellStyle(workbook))
        cell.setHyperlink(link)
      }
    }
    for (i <- reportInfo.indices) {
      sheet.autoSizeColumn(i)
    }
  }

  private def createDashboardReportTab(workbook: XSSFWorkbook, dashboardReport: util.Collection[StatusOverviewReportLine]): Unit = {
    val sheet = workbook.createSheet("Dashboard Info")
    val dashboardReportColumn = Array("Success", "Rollback", "Retried", "Aborted", "Total")
    val headerRow = sheet.createRow(0)
    headerRow.createCell(0).setCellValue("")
    for (i <- dashboardReportColumn.indices) {
      val cell = headerRow.createCell(i + 1)
      cell.setCellValue(dashboardReportColumn(i))
    }
    val countRow = sheet.createRow(1)
    countRow.createCell(0).setCellValue("Count")
    for (i <- dashboardReportColumn.indices) {
      countRow.createCell(i + 1)
    }
    val percentageRow = sheet.createRow(2)
    percentageRow.createCell(0).setCellValue("Percentage")
    var total = 0
    for ((p, _) <- dashboardReport.asScala.zipWithIndex) {
       p.executionState match {
         case "SUCCESS" => countRow.getCell(1).setCellValue(p.amount)
         case "ROLLBACK" => countRow.getCell(2).setCellValue(p.amount)
         case "FAILED"  => countRow.getCell(3).setCellValue(p.amount)
         case "ABORTED"  => countRow.getCell(4).setCellValue(p.amount)
         case "TOTAL" =>
           countRow.getCell(5).setCellValue(p.amount)
           total = p.amount
       }
    }
    for (i <- dashboardReportColumn.indices) {
      val cell = percentageRow.createCell(i + 1)
      val amount = countRow.getCell(i+1).getNumericCellValue
      cell.setCellValue(getFormattedPercentage(amount, total))
    }

    for (i <- 0 to 2) {
      sheet.autoSizeColumn(i)
    }
  }

  protected def getPercentage(numOfTasksInAState: Double, totalTasks: Double): Double = {
    checkArgument(numOfTasksInAState <= totalTasks, "Number of deplyments in a " +
      "state passed to the percentage function cannot be greater than total number of " +
      "deployments! numOfDeployment :" + numOfTasksInAState + "totalDeployment:" + totalTasks)
    if (totalTasks == 0) 0
    else (numOfTasksInAState / totalTasks) * 100
  }

  protected def getFormattedPercentage(numOfTasksInAState: Double, totalTasks: Double): String = {
    val df = new DecimalFormat("#.##", DecimalFormatSymbols.getInstance(Locale.ENGLISH))
    df.format(getPercentage(numOfTasksInAState, totalTasks)) + "%"
  }

  private def createDeploymentReportTab(workbook: XSSFWorkbook,
                                        deploymentReport: util.Collection[StatusOverviewDataReportLine]): Unit = {
    val sheet = workbook.createSheet("Deployment Info")
    sheet.setDefaultColumnWidth(30)
    val deploymentReportColumn = Array("Deployment Start DateTime", "Deployment End DateTime", "Deployment Status",
      "Task ID")
    val headerRow = sheet.createRow(0)
    for (i <- deploymentReportColumn.indices) {
      val cell = headerRow.createCell(i)
      cell.setCellValue(deploymentReportColumn(i))
    }
    for ((p, index) <- deploymentReport.asScala.zipWithIndex) {
      val row = sheet.createRow(index + 1)
      row.createCell(0).setCellValue(format("%s", DEPLOYMENT_DATE_FORMAT_FILE_REPORT.print(p.startDate)))
      row.createCell(1).setCellValue(format("%s", DEPLOYMENT_DATE_FORMAT_FILE_REPORT.print(p.endDate)))
      row.createCell(2).setCellValue(p.status)
      row.createCell(3).setCellValue(p.taskId)
    }
  }
}
