package com.xebialabs.deployit.core.util

import ai.digital.deploy.settings.GeneralSettings
import com.xebialabs.deployit.checks.Checks
import org.apache.tika.Tika

import java.util.Base64

object GeneralSettingsUtils {

  // 1 MB
  private val MAX_LOGO_FILE_SIZE_BYTES = 1048576
  private val MAX_INSTANCE_NAME_SIZE = 150

  private val tika = new Tika()
  private val base64 = "base64,"

  def validate(generalSetting: GeneralSettings): Unit = {
    validateInstanceNameLength(generalSetting.instanceName)
    validateLogoFile(generalSetting.logoFile)
  }

  private def validateInstanceNameLength(instanceName: String): Unit = {
    if(instanceName != null) {
      Checks.checkTrue(instanceName.length <= MAX_INSTANCE_NAME_SIZE, s"Instance name length should be maximum $MAX_INSTANCE_NAME_SIZE characters.")
    }
  }

  private def validateLogoFile(logoFile: String): Unit = {
    if (logoFile != null && logoFile.nonEmpty) {
      validateLogoFileSize(logoFile)
      validateLogoFileType(logoFile)
      validateLogoMimeType(logoFile)
    }
  }

  private def validateLogoFileSize(logoFile: String): Unit = {
    val characterCount = logoFile.length;
    val paddingCount = logoFile.substring(characterCount - 2, characterCount).count(c => c == '=')
    val sizeInBytes = (3 * (characterCount / 4)) - paddingCount;
    Checks.checkTrue(sizeInBytes <= MAX_LOGO_FILE_SIZE_BYTES, "Logo file is too large. Maximum size is 1 MB")
  }

  private def validateLogoFileType(logoFile: String): Unit = {
    Checks.checkTrue(logoFile.substring("data:".length).startsWith("image/"), "Logo file must be of an image file type.")
  }

  private def validateLogoMimeType(logoFile: String): Unit = {
    val decodedBytes = Base64.getDecoder.decode(logoFile.substring(logoFile.indexOf(base64)+base64.length))
    val detectedMimeType = tika.detect(decodedBytes)
    Checks.checkTrue(detectedMimeType.startsWith("image/"), "Logo file must be of an image file type.")
  }
}
