package com.xebialabs.deployit.core.rest.api.reports

import com.xebialabs.deployit.report.audit.{AuditPermissionRoleRow, RolePrincipalPermissionRow}
import com.xebialabs.deployit.security.Permissions
import org.apache.poi.common.usermodel.HyperlinkType
import org.apache.poi.ss.usermodel._
import org.apache.poi.xssf.usermodel.{XSSFFont, XSSFWorkbook}
import org.joda.time.LocalDateTime
import org.joda.time.format.{DateTimeFormat, DateTimeFormatter}

import java.lang.String.format
import java.util
import scala.jdk.CollectionConverters._

object AuditReportUtility {
  private val AUDIT_DATE_FORMAT_FILE_REPORT: DateTimeFormatter = DateTimeFormat.forPattern("MMMM d, yyyy HH:mm a")

  def createAuditReport(globalAuditReport: util.List[RolePrincipalPermissionRow],
                        folderAuditReport: util.List[AuditPermissionRoleRow],
                        serverName: String,
                        versionNumber: String): Workbook = {
    val workbook = new XSSFWorkbook
    createTemplateTab(workbook, serverName, versionNumber)
    createGlobalAuditReportTab(workbook, globalAuditReport)
    createFolderAuditReportTab(workbook, folderAuditReport)
    workbook
  }

  private def createTemplateTab(workbook: XSSFWorkbook, serverName: String, versionNumber: String): Unit = {
    val userName = Permissions.getAuthenticatedUserName
    val dateTime = format("%s", AUDIT_DATE_FORMAT_FILE_REPORT.print(LocalDateTime.now))
    val sheet = workbook.createSheet("Report Info")
    sheet.setDefaultRowHeightInPoints(19)
    sheet.setDisplayGridlines(false)
    val imageData = this.getClass.getResource("/xl_deploy_logo_green.png")
    ExcelElements.insertImage(imageData, workbook, sheet)
    val cellStyle = ExcelStyle.getCellStyle(workbook)
    val cellBoldStyle = ExcelStyle.getBoldCellStyle(workbook)
    val reportInfo = Array("Generated On", "Generated By", "XL Deploy instance")
    val reportInfoValue = Array(dateTime, userName, serverName)

    val firstRow = sheet.createRow(0)
    firstRow.createCell(0).setCellValue("XL Deploy User Permissions Report")
    firstRow.getCell(0).setCellStyle(cellStyle)
    ExcelElements.addCell(firstRow, workbook)

    sheet.createRow(1).createCell(0).setCellValue("")

    val row2 = sheet.createRow(2)
    row2.createCell(0).setCellValue("Report Info")
    row2.getCell(0).setCellStyle(cellBoldStyle)

    for (i <- 3 to 5) {
      val row = sheet.createRow(i - 1)
      row.createCell(1).setCellValue(reportInfo(i - 3))
      row.createCell(2).setCellValue(reportInfoValue(i - 3))

      if (i == 5) {
        val createHelper = workbook.getCreationHelper
        val link = createHelper.createHyperlink(HyperlinkType.URL)
        val hLinkFont: XSSFFont = workbook.createFont
        val hLinkStyle = workbook.createCellStyle
        hLinkFont.setUnderline(Font.U_SINGLE)
        hLinkStyle.setFont(hLinkFont)
        link.setAddress(serverName)
        val cell = row.getCell(2)
        cell.setCellStyle(ExcelStyle.getHyperlinkCellStyle(workbook))
        cell.setHyperlink(link)
      }
    }

    sheet.createRow(5).createCell(7).setCellValue(versionNumber)
    for (i <- reportInfo.indices) {
      sheet.autoSizeColumn(i)
    }
  }

  private def createGlobalAuditReportTab(workbook: XSSFWorkbook, globalAuditReport: util.List[RolePrincipalPermissionRow]): Unit = {
    val sheet = workbook.createSheet("Global")
    sheet.setDisplayGridlines(false)
    sheet.setDefaultColumnWidth(30)
    val globalAuditReportColumn = Array("Roles", "Principals", "Permissions")
    val cellStyle = ExcelStyle.getCellStyle(workbook)
    val headerRow = sheet.createRow(0)
    for (i <- globalAuditReportColumn.indices) {
      val cell = headerRow.createCell(i)
      cell.setCellValue(globalAuditReportColumn(i))
      cell.setCellStyle(cellStyle)
    }
    val cellBorderStyle = ExcelStyle.cellBorderStyle(workbook)
    for ((p, index) <- globalAuditReport.asScala.zipWithIndex) {
      val row = sheet.createRow(index + 1)
      var cell = row.createCell(0)
      cell.setCellValue(p.role)
      cell.setCellStyle(cellBorderStyle)
      cell = row.createCell(1)
      cell.setCellValue(p.principal)
      cell.setCellStyle(cellBorderStyle)
      cell = row.createCell(2)
      cell.setCellValue(p.permission)
      cell.setCellStyle(cellBorderStyle)
    }
  }

  private def createFolderAuditReportTab(workbook: XSSFWorkbook, folderAuditReport: util.List[AuditPermissionRoleRow]): Unit = {
    val sheet = workbook.createSheet("Folder")
    sheet.setDisplayGridlines(false)
    sheet.setDefaultColumnWidth(30)
    val folderAuditReportColumn = Array("Folder", "Folder Permissions", "Role")
    val cellStyle = ExcelStyle.getCellStyle(workbook)
    val headerRow = sheet.createRow(0)
    for (i <- folderAuditReportColumn.indices) {
      val cell = headerRow.createCell(i)
      cell.setCellValue(folderAuditReportColumn(i))
      cell.setCellStyle(cellStyle)
    }
    val creationHelper = workbook.getCreationHelper
    val cellBorderStyle = ExcelStyle.cellBorderStyle(workbook)
    for ((f, index) <- folderAuditReport.asScala.zipWithIndex) {
      val row = sheet.createRow(index + 1)
      for (_ <- folderAuditReportColumn.indices) {
        var cell = row.createCell(0)
        cell.setCellValue(f.folder)
        cell.setCellStyle(cellBorderStyle)
        cell = row.createCell(1)
        cell.setCellValue(f.permission)
        cell.setCellStyle(cellBorderStyle)
        cell = row.createCell(2)
        cell.setCellValue(f.role)
        cell.setCellStyle(cellBorderStyle)
        val link = creationHelper.createHyperlink(HyperlinkType.DOCUMENT)
        link.setAddress("'Global'!A1")
        cell.setHyperlink(link)
      }
    }
  }
}
