package com.xebialabs.deployit.core.service.status

import ai.digital.deploy.webhook.WebhookServer
import com.xebialabs.deployit.ServerConfiguration
import com.xebialabs.deployit.core.events.dto.{ApplicationDeploymentPackageState, DeployedApplicationPackage}
import com.xebialabs.deployit.core.service.ApplicationAndEnvironmentFilter
import com.xebialabs.deployit.engine.spi.exception.DeployitException
import com.xebialabs.deployit.plugin.api.udm.{Application, Environment}
import com.xebialabs.deployit.repository.RepositoryService
import com.xebialabs.deployit.service.dependency.RepositoryServiceAware
import com.xebialabs.deployit.util.PasswordEncrypter
import grizzled.slf4j.Logging
import org.apache.hc.client5.http.config.RequestConfig
import org.apache.hc.client5.http.impl.auth.SystemDefaultCredentialsProvider
import org.apache.hc.client5.http.impl.classic.HttpClients
import org.apache.hc.client5.http.impl.io.PoolingHttpClientConnectionManagerBuilder
import org.apache.hc.client5.http.impl.routing.SystemDefaultRoutePlanner
import org.apache.hc.client5.http.ssl.{NoopHostnameVerifier, SSLConnectionSocketFactoryBuilder}
import org.apache.http.ssl.SSLContextBuilder
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.boot.web.client.RestTemplateBuilder
import org.springframework.context.annotation.{Bean, Configuration}
import org.springframework.http.HttpStatus.Series
import org.springframework.http.client.{ClientHttpResponse, HttpComponentsClientHttpRequestFactory}
import org.springframework.http.{HttpEntity, HttpHeaders, HttpMethod, HttpStatus}
import org.springframework.stereotype.Service
import org.springframework.util.LinkedMultiValueMap
import org.springframework.web.client.{ResponseErrorHandler, RestTemplate}

import java.io.File
import java.net.ProxySelector
import java.security.cert.X509Certificate
import java.time.Duration
import java.util.concurrent.{CompletableFuture, TimeUnit}
import scala.jdk.CollectionConverters._

trait DeploymentPackageStatusService {
  def send(state: ApplicationDeploymentPackageState): Unit

  def send(deployedApplicationPackage: DeployedApplicationPackage): Unit
}

@Configuration
class WebhookDeploymentPackageStatusConfiguration(serverConfiguration: ServerConfiguration) {
  def webhookRestTemplateResponseErrorHandler(): ResponseErrorHandler =
    new ResponseErrorHandler() {
      override def hasError(httpResponse: ClientHttpResponse): Boolean =
        HttpStatus.valueOf(httpResponse.getStatusCode.value()).series() == Series.CLIENT_ERROR ||
          HttpStatus.valueOf(httpResponse.getStatusCode.value()).series() == Series.SERVER_ERROR

      override def handleError(response: ClientHttpResponse): Unit =
        HttpStatus.valueOf(response.getStatusCode.value()).series() match {
          case Series.SERVER_ERROR =>
            throw DeploymentPackageStatusServerException(HttpStatus.valueOf(response.getStatusCode.value()).getReasonPhrase)
          case Series.CLIENT_ERROR =>
            throw DeploymentPackageStatusClientException(HttpStatus.valueOf(response.getStatusCode.value()).getReasonPhrase)
        }
    }

  @Bean
  def webhookServiceRestTemplate(builder: RestTemplateBuilder): RestTemplate = {
    val sslContextBuilder = SSLContextBuilder.create()

    val timeout = Duration.ofSeconds(1).toMillis.toInt
    if (serverConfiguration.isSsl)
      Option(serverConfiguration.getKeyStorePath)
        .map(path =>
          sslContextBuilder.loadTrustMaterial(new File(path),
            Option(ServerConfiguration.getInstance().getKeyStorePassword)
              .map(PasswordEncrypter.getInstance().ensureDecrypted).map(_.toCharArray).orNull
          )
        )
        .getOrElse(sslContextBuilder.loadTrustMaterial(null, (_: Array[X509Certificate], _: String) => true))
    else
      sslContextBuilder.loadTrustMaterial(null, (_: Array[X509Certificate], _: String) => true)

    val connectionManager = PoolingHttpClientConnectionManagerBuilder
      .create
      .setSSLSocketFactory(
        SSLConnectionSocketFactoryBuilder.create()
          .setSslContext(sslContextBuilder.build())
          .setHostnameVerifier(NoopHostnameVerifier.INSTANCE)
          .build()
      )
    val httpClientBuilder = HttpClients.custom
      .setConnectionManager(connectionManager.build())
      .setRoutePlanner(new SystemDefaultRoutePlanner(ProxySelector.getDefault))
      .setDefaultCredentialsProvider(new SystemDefaultCredentialsProvider)
      .setDefaultRequestConfig(RequestConfig.custom
        .setResponseTimeout(timeout, TimeUnit.MILLISECONDS)
        .setConnectTimeout(timeout, TimeUnit.MILLISECONDS)
        .build)

    builder
      .requestFactory(() => new HttpComponentsClientHttpRequestFactory(httpClientBuilder.build()))
      .errorHandler(webhookRestTemplateResponseErrorHandler())
      .build()
    builder.build()
  }
}

@Service
class WebhookDeploymentPackageStatusClient @Autowired()(
                                                         val repositoryService: RepositoryService,
                                                         @Qualifier("webhookServiceRestTemplate") webhookServiceRestTemplate: RestTemplate
                                                       )
  extends DeploymentPackageStatusService with RepositoryServiceAware with ApplicationAndEnvironmentFilter with Logging {
  final val ContentType = "application/json"

  override def send(state: ApplicationDeploymentPackageState): Unit =
    executeRequest(state, state.applicationUid, state.state.destinationUid)

  override def send(deployedApplicationPackage: DeployedApplicationPackage): Unit =
    executeRequest(deployedApplicationPackage, deployedApplicationPackage.applicationUid, deployedApplicationPackage.destinationUid)

  private def getServers: Seq[WebhookServer] =
    repositoryService.
      listEntities[WebhookServer](typedSearchParameters[WebhookServer]).asScala.filter(_.enabled).toVector

  private def executeRequest[T](requestBody: T, applicationUid: String, environmentUid: String): Unit = {
    val appId = repositoryService.readByReferenceId[Application](applicationUid).getId
    val envId = repositoryService.readByReferenceId[Environment](environmentUid).getId

    getServers.filter { server =>
      matchAppEnvToDeploymentRegexp(server, appId, envId)
    }.foreach(sendAsync(_, requestBody))
  }

  private def sendAsync[T](server: WebhookServer, requestBody: T) = {
    CompletableFuture.runAsync(() => {
      val headers = new LinkedMultiValueMap[String, String]()
      if (!server.noAuthentication) {
        headers.add(server.tokenHeader, server.tokenValue)
      }
      headers.add(HttpHeaders.CONTENT_TYPE, ContentType)
      webhookServiceRestTemplate.exchange(
        server.webhookServerURL,
        HttpMethod.POST,
        new HttpEntity[T](requestBody, headers),
        classOf[Void]
      )
    }).whenComplete((_, ex) => {
      Option(ex) match {
        case Some(exception) =>
          logger.warn(s"An error occurred when sending webhook event to ${server.webhookServerURL}: ${exception.getMessage}")
      }
    })
  }
}

final case class DeploymentPackageStatusServerException(message: String) extends DeployitException(message)
final case class DeploymentPackageStatusClientException(message: String) extends DeployitException(message)
