package com.xebialabs.deployit.core.rest.converters;

import org.springframework.messaging.Message;
import org.springframework.messaging.MessageHeaders;
import org.springframework.messaging.converter.SmartMessageConverter;
import org.springframework.messaging.support.MessageBuilder;
import org.springframework.util.MimeType;

import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.ext.MessageBodyWriter;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;

public abstract class AbstractMessageConverter<T> implements SmartMessageConverter {

    private final MessageBodyWriter<T> messageBodyWriter;

    private static final int DEFAULT_SIZE = 1024 * 1024;

    private final MediaType mediaType = MediaType.valueOf(MediaType.APPLICATION_JSON);

    private final MimeType mimeType = new MimeType("application", "json", StandardCharsets.UTF_8);

    AbstractMessageConverter(MessageBodyWriter<T> messageBodyWriter) {
        this.messageBodyWriter = messageBodyWriter;
    }

    @Override
    public Object fromMessage(Message<?> message, Class<?> targetClass, Object conversionHint) {
        return null;
    }

    @Override
    public Message<?> toMessage(Object payload, MessageHeaders headers, Object conversionHint) {
        if (!canConvertTo(payload)) {
            return null;
        }

        ByteArrayOutputStream baos = new ByteArrayOutputStream(DEFAULT_SIZE);
        try {
            messageBodyWriter.writeTo((T) payload, payload.getClass(),
                    payload.getClass().getGenericSuperclass(), null, mediaType, null, baos);
        } catch (IOException e) {
            return null;
        }
        return buildMessage(headers, baos);
    }

    private Message<?> buildMessage(MessageHeaders headers, ByteArrayOutputStream baos) {
        MessageBuilder<?> builder = MessageBuilder.withPayload(baos.toByteArray());
        if (headers != null) {
            builder.copyHeaders(headers);
        }
        builder.setHeaderIfAbsent(MessageHeaders.CONTENT_TYPE, mimeType);
        return builder.build();
    }

    @Override
    public Object fromMessage(Message<?> message, Class<?> targetClass) {
        return null;
    }

    @Override
    public Message<?> toMessage(Object payload, MessageHeaders headers) {
        return toMessage(payload, headers, null);
    }

    boolean canConvertTo(Object payload) {
        return messageBodyWriter.isWriteable(payload.getClass(), payload.getClass().getGenericSuperclass(), null, mediaType);
    }
}
