package com.xebialabs.deployit.core.rest.json;

import com.google.common.base.Charsets;
import com.google.common.io.CharStreams;
import com.xebialabs.deployit.core.xml.PasswordEncryptingCiConverter;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.xltype.serialization.CiWriter;
import com.xebialabs.xltype.serialization.json.CiJsonReader;
import com.xebialabs.xltype.serialization.json.CiJsonWriter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import jakarta.ws.rs.Consumes;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.WebApplicationException;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.MultivaluedMap;
import jakarta.ws.rs.ext.MessageBodyReader;
import jakarta.ws.rs.ext.MessageBodyWriter;
import jakarta.ws.rs.ext.Provider;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.lang.annotation.Annotation;
import java.lang.reflect.Type;

@Component
@Provider
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
public class CiJsonMessageBodyReaderWriter implements MessageBodyWriter<ConfigurationItem>, MessageBodyReader<ConfigurationItem> {
    private final ReaderPostProcessor postProcessor;

    @Autowired
    public CiJsonMessageBodyReaderWriter(ReaderPostProcessor postProcessor) {
        this.postProcessor = postProcessor;
    }

    @Override
    public boolean isWriteable(Class<?> type, Type genericType, Annotation[] annotations, MediaType mediaType) {
        return isReadable(type, genericType, annotations, mediaType);
    }

    @Override
    public boolean isReadable(Class<?> type, Type genericType, Annotation[] annotations, MediaType mediaType) {
        return mediaType.toString().contains(MediaType.APPLICATION_JSON) && ConfigurationItem.class.isAssignableFrom(type);
    }

    @Override
    public long getSize(ConfigurationItem t, Class<?> type, Type genericType, Annotation[] annotations, MediaType mediaType) {
        return -1L;
    }

    @Override
    public void writeTo(ConfigurationItem ci, Class<?> type, Type genericType, Annotation[] annotations, MediaType mediaType, MultivaluedMap<String, Object> httpHeaders,
                        OutputStream entityStream) throws IOException, WebApplicationException {
        PasswordEncryptingCiConverter converter = getConverter();
        CiWriter writer = new CiJsonWriter();
        converter.writeCi(ci, writer);

        entityStream.write(writer.toString().getBytes());
    }

    @Override
    public ConfigurationItem readFrom(Class<ConfigurationItem> type, Type genericType, Annotation[] annotations, MediaType mediaType, MultivaluedMap<String, String> httpHeaders,
                                      InputStream entityStream) throws WebApplicationException {
        boolean skipNotExistingCis = "true".equals(httpHeaders.getFirst("skipNotExistingCis"));
        return postProcessor.apply(context -> {
            PasswordEncryptingCiConverter converter = getConverter();
            context.register(converter);
            return converter.readCi(CiJsonReader.create(toString(entityStream)));
        }, skipNotExistingCis, Integer.MAX_VALUE);
    }

    private PasswordEncryptingCiConverter getConverter() {
        return new PasswordEncryptingCiConverter();
    }

    public static String toString(InputStream in) {
        try {
            return CharStreams.toString(new InputStreamReader(in, Charsets.UTF_8));
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }
}
