package com.xebialabs.deployit.core.rest.api

import ai.digital.deploy.sql.model.{AddUserProfile, UserProfileView}
import com.xebialabs.deployit.core.api.XldUserProfileService
import com.xebialabs.deployit.core.rest.api.support.PaginationSupport
import com.xebialabs.deployit.engine.api.dto
import com.xebialabs.deployit.engine.api.dto.Paging
import com.xebialabs.deployit.security.model.XldUserProfile
import com.xebialabs.deployit.security.permission.PlatformPermissions.{EDIT_SECURITY, VIEW_SECURITY}
import com.xebialabs.deployit.security.service.UserProfileService
import jakarta.ws.rs.core.Context
import org.jboss.resteasy.spi.HttpResponse
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Controller

import java.{lang, util}
import scala.jdk.CollectionConverters._

@Controller
class XldUserProfileServiceImpl(@Autowired userProfileService: UserProfileService)
  extends AbstractUserRestrictedResource with XldUserProfileService {

  @Context private val response: HttpResponse = null

  override def getUserProfile(username: String, loadCredentials: lang.Boolean): UserProfileView = {
    checkCurrentUserOrPermissions(username, VIEW_SECURITY, EDIT_SECURITY)
    val profile = userProfileService.findOne(username, loadCredentials)
    new UserProfileView(
      profile.analyticsEnabled,
      profile.credentials.map(credentials => XldUserCredentialsService.mapToUserCredentialsView(credentials)).asJava,
      profile.username
    )
  }

  override def createOrUpdateUserProfile(username: String, userProfile: AddUserProfile): Unit = {
    checkCurrentUserOrPermission(username, EDIT_SECURITY)
    userProfileService.createOrUpdate(username, userProfile.getAnalyticsEnabled)
  }

  override def listAllUserProfiles(username: String, fullName: String, email: String, paging: Paging, order: dto.Ordering): util.List[XldUserProfile] = {
    checkPermissions(EDIT_SECURITY, VIEW_SECURITY)
    val dataList = userProfileService.listAllUserProfiles(username, fullName, email, paging, order)
    PaginationSupport.addTotalCountHeader(userProfileService.countUsers(username, fullName, email), response)
    dataList
  }
}
