package com.xebialabs.deployit.core.rest.api

import java.util

import ai.digital.deploy.sql.http.enricher.PaginationService
import com.xebialabs.deployit.core.api.EnvironmentService
import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource
import com.xebialabs.deployit.engine.api.dto.{ConfigurationItemId, DeploymentInfo, Ordering, Paging}
import com.xebialabs.deployit.repository.{DeployedApplicationsRepository, HostRepository}
import grizzled.slf4j.Logging
import jakarta.ws.rs.core.Context
import org.jboss.resteasy.spi.HttpResponse
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import scala.jdk.CollectionConverters._

@Component
class EnvironmentServiceImpl @Autowired()(deployedApplicationsRepository: DeployedApplicationsRepository,
                                          hostRepository: HostRepository,
                                          paginationService: PaginationService,
                                          repositoryResource: RepositoryResource)
  extends AbstractSecuredResource with EnvironmentService with Logging {
  @Context val response: HttpResponse = null

  private def checkIfUserHasPermissions(ciId: String) =
    repositoryResource.read(ciId)

  override def getDeployedApplicationsByEnvironment(envId: String, deployedAppName: String, paging: Paging, order: Ordering): util.List[ConfigurationItemId] = {
    checkIfUserHasPermissions(envId)
    val limited = paginationService.getLimitedPaging(paging)
    paginationService.addPagingHeaderIfNeeded(paging, limited, paginationService.toSetHeader(response), () => deployedApplicationsRepository.countByEnvironment(envId, deployedAppName))
    deployedApplicationsRepository.findByEnvironment(envId, deployedAppName, paging, order).asJava
  }

  override def getHostsByEnvironment(envId: String, hostName: String, paging: Paging, order: Ordering): util.List[DeploymentInfo] = {
    checkIfUserHasPermissions(envId)
    val limited = paginationService.getLimitedPaging(paging)
    paginationService.addPagingHeaderIfNeeded(paging, limited, paginationService.toSetHeader(response), () => hostRepository.countByEnvironment(envId, hostName))
    hostRepository.findByEnvironment(envId, hostName, paging, order).asJava
  }

}
