package com.xebialabs.deployit.core.rest.json;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Charsets;
import com.google.common.io.CharStreams;
import com.xebialabs.deployit.core.util.ReaderWriterUtils;
import com.xebialabs.deployit.core.xml.PasswordEncryptingCiConverter;
import com.xebialabs.deployit.engine.api.dto.Deployment;
import com.xebialabs.xltype.serialization.json.DeploymentJsonConverter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import jakarta.ws.rs.Consumes;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.WebApplicationException;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.MultivaluedMap;
import jakarta.ws.rs.ext.MessageBodyReader;
import jakarta.ws.rs.ext.MessageBodyWriter;
import jakarta.ws.rs.ext.Provider;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.lang.annotation.Annotation;
import java.lang.reflect.Type;

@Component
@Provider
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
public class DeploymentJsonReaderWriter implements MessageBodyWriter<Deployment>, MessageBodyReader<Deployment> {
    private final ReaderPostProcessor postProcessor;

    @Autowired
    public DeploymentJsonReaderWriter(ReaderPostProcessor postProcessor) {
        this.postProcessor = postProcessor;
    }

    @Override
    public boolean isWriteable(Class<?> type, Type genericType, Annotation[] annotations, MediaType mediaType) {
        return isReadable(type, genericType, annotations, mediaType);
    }

    @Override
    public boolean isReadable(Class<?> type, Type genericType, Annotation[] annotations, MediaType mediaType) {
        return Deployment.class.isAssignableFrom(type);
    }

    @Override
    public long getSize(final Deployment deployment, final Class<?> type, final Type genericType, final Annotation[] annotations, final MediaType mediaType) {
        return -1L;
    }

    @Override
    public void writeTo(final Deployment deployment, final Class<?> type, final Type genericType, final Annotation[] annotations, final MediaType mediaType, final MultivaluedMap<String, Object> httpHeaders, final OutputStream entityStream) throws IOException, WebApplicationException {
        entityStream.write(toJson(deployment).getBytes());
    }

    @Override
    public Deployment readFrom(Class<Deployment> type, Type genericType, Annotation[] annotations, MediaType mediaType, MultivaluedMap<String, String> httpHeaders, InputStream entityStream) throws IOException, WebApplicationException {
        return toDeployment(toString(entityStream), ReaderWriterUtils.getCiTreeDepth(annotations));
    }

    @VisibleForTesting
    String toJson(final Deployment deployment) {
        DeploymentJsonConverter converter = getConverter();
        return converter.toJson(deployment);
    }

    @VisibleForTesting
    Deployment toDeployment(final String deployment, Integer depth) {
        return postProcessor.apply(context -> {
            DeploymentJsonConverter converter = getConverter();
            context.register(converter.getCiConverter());
            return converter.toDeployment(deployment);
        }, depth);
    }

    public String toString(InputStream in) throws IOException {
        return CharStreams.toString(new InputStreamReader(in, Charsets.UTF_8));
    }

    private DeploymentJsonConverter getConverter() {
        return new DeploymentJsonConverter(new PasswordEncryptingCiConverter());
    }
}
