package com.xebialabs.deployit.core.rest.api

import java.util

import ai.digital.deploy.tasker.common.TaskMetadata
import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource
import com.xebialabs.deployit.engine.api.QueueService
import com.xebialabs.deployit.engine.api.dto.{QueuedTask, TaskQueueSummary}
import com.xebialabs.deployit.engine.api.execution.{FetchMode, TaskWithBlock}
import com.xebialabs.deployit.engine.tasker.TaskExecutionEngine
import com.xebialabs.deployit.engine.tasker.repository.PendingTaskRepository
import com.xebialabs.deployit.security.permission.PlatformPermissions.ADMIN
import com.xebialabs.deployit.spring.BeanWrapper
import com.xebialabs.xltype.serialization.util.DateUtil
import grizzled.slf4j.Logging
import org.joda.time.{DateTime, Seconds}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import scala.jdk.CollectionConverters._

@Service
@Autowired
class QueueServiceImpl(engine: BeanWrapper[TaskExecutionEngine],
                       pendingTaskRepository: PendingTaskRepository)
  extends AbstractSecuredResource with QueueService with Logging {

  override def queuedTasks(): util.List[QueuedTask] = {
    checkPermission(ADMIN)
    val queuedTasks = pendingTaskRepository.queuedTasks().asScala.zipWithIndex.map {
      case (task, index) =>
        val satellitesString = task.metadata.get(TaskMetadata.SATELLITE_IDS)
        val satellites = if (satellitesString != null && !satellitesString.isBlank) satellitesString.split(",").map(_.trim).toList else List()
        new QueuedTask(
          task.id,
          task.metadata.get(TaskMetadata.TASK_TYPE),
          DateUtil.toString(task.queuedDate),
          index + 1,
          satellites.asJava,
          task.metadata.get(TaskMetadata.ENVIRONMENT_ID))
    }
    queuedTasks.asJava
  }

  override def taskQueueSummary(): TaskQueueSummary = {
    checkPermission(ADMIN)
    val statePredicate = (task: TaskWithBlock) => task.getState != null && task.getState.isExecutingSteps
    val numOfRunningTasks = engine.get().getAllIncompleteTasks(FetchMode.SUMMARY).asScala.count(statePredicate)
    val queuedTasks = pendingTaskRepository.queuedTasks().asScala
    val diffInSeconds = if (queuedTasks.nonEmpty) Seconds.secondsBetween(queuedTasks.headOption.orNull.queuedDate, DateTime.now()) else Seconds.ZERO
    new TaskQueueSummary(
      numOfRunningTasks,
      queuedTasks.length,
      diffInSeconds.getSeconds
    )
  }
}
