package com.xebialabs.deployit.core.rest.api

import ai.digital.deploy.sql.http.enricher.PaginationService
import ai.digital.deploy.sql.model._
import com.xebialabs.deployit.booter.local.utils.Strings.isNotBlank
import com.xebialabs.deployit.core.api.PlaceholderService
import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource
import com.xebialabs.deployit.engine.api.dto
import com.xebialabs.deployit.engine.api.dto.Paging
import com.xebialabs.deployit.plugin.api.deployment._
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.repository.placeholders.{PlaceholderRepository, ResolvedPlaceholderRepository}
import com.xebialabs.deployit.security.permission.PermissionHelper
import com.xebialabs.deployit.security.permission.PlatformPermissions.READ
import com.xebialabs.deployit.security.{RolesPermissionsPair, SecurityServiceLocator}
import org.jboss.resteasy.spi.HttpResponse
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import java.util
import javax.ws.rs.core.Context
import scala.jdk.CollectionConverters._

@Service
class PlaceholderServiceImpl(@Autowired resolvedPlaceholderRepository: ResolvedPlaceholderRepository,
                             @Autowired paginationService: PaginationService,
                             @Autowired placeholderRepository: PlaceholderRepository) extends AbstractSecuredResource with PlaceholderService {

  @Context val response: HttpResponse = null

  private def pathToId(path: String): String = path.substring(1)

  private def toDefinedPlaceholderView(placeholders: List[Placeholder]): List[DefinedPlaceholderView] = {
    val canReadCi = (placeholder: Placeholder) => hasReadPermission(placeholder.ciPath)

    placeholders.foldLeft(List[DefinedPlaceholderView]())((acc: List[DefinedPlaceholderView], placeholder: Placeholder) =>
      acc.::(new DefinedPlaceholderView(
        placeholder.key, pathToId(placeholder.ciPath), placeholder.ciType.toString, canReadCi(placeholder)
      ))
    ).reverse
  }

  def isEncrypted(placeholder: PlaceholderCiWithValue): Boolean = {
    Option(placeholder.encryptedValue).isDefined ||
      placeholder.ciType.instanceOf(Type.valueOf("udm.EncryptedDictionary")) ||
      placeholder.ciType.isSubTypeOf(Type.valueOf("udm.EncryptedDictionary"))
  }

  private def toDefinedPlaceholderWithDictionaryView(placeholders: List[PlaceholderCiWithValue]): List[DictionaryWithValueView] = {
    placeholders.foldLeft(List[DictionaryWithValueView]())((acc: List[DictionaryWithValueView], placeholder: PlaceholderCiWithValue) => {
      val encrypted = isEncrypted(placeholder)
      val id = pathToId(placeholder.relatedCiId)
      acc.::(
        new DictionaryWithValueView(
          if (encrypted || !hasReadPermission(id)) "" else placeholder.value, encrypted,
          new DefinedPlaceholderReference(id, placeholder.ciType.toString)
        ))
    }
    ).reverse
  }

  private def toDefinedPlaceholderWithEnvironmentView(placeholders: List[PlaceholderCi]): List[DefinedPlaceholderReference] = {
    placeholders.foldLeft(List[DefinedPlaceholderReference]())((acc: List[DefinedPlaceholderReference], placeholder: PlaceholderCi) => {
      val id = pathToId(placeholder.relatedCiId)
      acc.::(
        new DefinedPlaceholderReference(id, placeholder.ciType.toString)
      )
    }).reverse
  }

  def toPlaceholdersView(placeholders: List[ResolvedPlaceholder]): List[ResolvedPlaceholderView] = {
    val canReadDictionary = (placeholder: ResolvedPlaceholder) => hasReadPermission(placeholder.dictionaryId)

    placeholders.foldLeft(List[ResolvedPlaceholderView]())((acc: List[ResolvedPlaceholderView], placeholder: ResolvedPlaceholder) =>
      acc.::(new ResolvedPlaceholderView(
        placeholder.isEncrypted,
        placeholder.key,
        if (canReadDictionary(placeholder) && placeholder.value != null) placeholder.value else "",
        new ResolvedPlaceholderReference(
          placeholder.containerId,
          hasReadPermission(placeholder.containerId),
          isNotBlank(placeholder.containerId) && !placeholder.containerDeleted
        ),
        new ResolvedPlaceholderReference(
          placeholder.deployedAppId,
          hasReadPermission(placeholder.deployedAppId),
          true
        ),
        new ResolvedPlaceholderReference(
          placeholder.dictionaryId,
          canReadDictionary(placeholder),
          !placeholder.dictionaryDeleted
        ),
        new ResolvedPlaceholderReference(
          placeholder.environmentId,
          hasReadPermission(placeholder.environmentId),
          !placeholder.environmentDeleted
        ),
        placeholder.versionId
      ))
    ).reverse
  }

  override def findPlaceholdersForEnvironment(environmentId: String,
                                              key: String,
                                              value: String,
                                              deployedName: String,
                                              dictionaryName: String,
                                              hostName: String,
                                              paging: Paging,
                                              order: dto.Ordering): util.List[ResolvedPlaceholderView] = {
    paginationService.addPagingHeaderIfNeeded(paginationService.toSetHeader(response), () =>
      resolvedPlaceholderRepository.countAllResolvedPlaceholdersForEnvironment(environmentId, Option(key),
        Option(value),
        Option(dictionaryName),
        Option(deployedName),
        Option(hostName)), paging)

    val placeholders = resolvedPlaceholderRepository.getResolvedPlaceholdersForEnvironment(environmentId, Option(key),
      Option(value),
      Option(dictionaryName),
      Option(deployedName),
      Option(hostName),
      paging,
      order)
    toPlaceholdersView(placeholders).asJava
  }

  override def findPlaceholdersForHost(hostId: String,
                                       key: String,
                                       value: String,
                                       deployedName: String,
                                       dictionaryName: String,
                                       environmentName: String,
                                       paging: Paging,
                                       order: dto.Ordering): util.List[ResolvedPlaceholderView] = {

    paginationService.addPagingHeaderIfNeeded(paginationService.toSetHeader(response), () =>
      resolvedPlaceholderRepository.countAllResolvedPlaceholdersForContainer(hostId, Option(key),
        Option(value),
        Option(dictionaryName),
        Option(deployedName),
        Option(environmentName)), paging)

    val placeholders = resolvedPlaceholderRepository.getResolvedPlaceholdersForContainer(hostId, Option(key),
      Option(value),
      Option(dictionaryName),
      Option(deployedName),
      Option(environmentName),
      paging,
      order)
    toPlaceholdersView(placeholders).asJava
  }

  override def findArchivedPlaceholdersForEnvironment(environmentId: String,
                                                      key: String,
                                                      value: String,
                                                      deployedName: String,
                                                      dictionaryName: String,
                                                      hostName: String,
                                                      taskId: String,
                                                      paging: Paging,
                                                      order: dto.Ordering): util.List[ResolvedPlaceholderView] = {
    paginationService.addPagingHeaderIfNeeded(paginationService.toSetHeader(response),
      () => resolvedPlaceholderRepository.countAllArchivedResolvedPlaceholdersForEnvironment(environmentId, Option(key),
        Option(value),
        Option(dictionaryName),
        Option(deployedName),
        Option(hostName),
        Option(taskId)), paging)

    val placeholders = resolvedPlaceholderRepository.getArchivedResolvedPlaceholdersForEnvironment(environmentId, Option(key),
      Option(value),
      Option(dictionaryName),
      Option(deployedName),
      Option(hostName),
      Option(taskId),
      paging,
      order)

    toPlaceholdersView(placeholders).asJava
  }

  override def definedPlaceholders(key: String,
                                   applicationId: String,
                                   applicationName: String,
                                   paging: Paging,
                                   order: dto.Ordering
                                  ): util.List[DefinedPlaceholderView] = {

    val rolesPermissionsPair = getRolesPermissions

    paginationService.addPagingHeaderIfNeeded(paginationService.toSetHeader(response),
      () => placeholderRepository.countAllPlaceholders(key, applicationId, applicationName, rolesPermissionsPair),
      paging)

    val placeholders = placeholderRepository.getPlaceholders(key, applicationId, applicationName, rolesPermissionsPair, paging, order)
    toDefinedPlaceholderView(placeholders).asJava
  }

  override def definedPlaceholdersNames(namePattern: String,
                                        paging: Paging,
                                        order: dto.Ordering
                                       ): util.List[String] = {
    placeholderRepository.getPlaceholderKeys(namePattern, None, paging, order).asJava
  }

  override def definedPlaceholdersByDictionary(key: String,
                                               value: String,
                                               dictionaryId: String,
                                               dictionaryName: String,
                                               paging: Paging,
                                               order: dto.Ordering): util.List[DictionaryWithValueView] = {

    val rolesPermissionsPair = getRolesPermissions
    paginationService.addPagingHeaderIfNeeded(paginationService.toSetHeader(response),
      () => placeholderRepository.countPlaceholdersWithDictionary(key, value, dictionaryId, dictionaryName, rolesPermissionsPair),
      paging)

    val placeholders = placeholderRepository.getPlaceholdersWithDictionary(key, value, dictionaryId, dictionaryName, rolesPermissionsPair, paging, order)
    toDefinedPlaceholderWithDictionaryView(placeholders).asJava
  }

  override def definedPlaceholdersByEnvironment(key: String,
                                                environmentId: String,
                                                environmentName: String,
                                                paging: Paging,
                                                order: dto.Ordering): util.List[DefinedPlaceholderReference] = {

    val rolesPermissionsPair = getRolesPermissions
    paginationService.addPagingHeaderIfNeeded(paginationService.toSetHeader(response),
      () => placeholderRepository.countPlaceholdersWithEnvironment(key, environmentId, environmentName, rolesPermissionsPair),
      paging)

    val placeholders = placeholderRepository.getPlaceholdersWithEnvironment(key, environmentId, environmentName, rolesPermissionsPair, paging, order)
    toDefinedPlaceholderWithEnvironmentView(placeholders).asJava
  }

  private def getRolesPermissions: Option[RolesPermissionsPair] =
    if (!PermissionHelper.isCurrentUserAdmin || (SecurityServiceLocator.getViewAsData != null && (SecurityServiceLocator.getViewAsData.getRoles != null || SecurityServiceLocator.getViewAsData.getUser != null))) {
      Option(READ.getPermissionHandler.getRolesPermissionsPair)
    } else {
      None
    }
}
