package com.xebialabs.deployit.core.service.impl

import ai.digital.deploy.task.status.util.DeploymentStatusUtils
import ai.digital.deploy.tasker.common.{TaskMetadata, TaskType}
import com.xebialabs.deployit.core.rest.api.RepositoryResource
import com.xebialabs.deployit.core.service.ApplicationStatusService
import com.xebialabs.deployit.engine.api.dto.{ApplicationDeploymentPackageState, DeploymentPackageState, Ordering, VersionTag}
import com.xebialabs.deployit.engine.api.execution.{FetchMode, TaskExecutionState, TaskWithBlock}
import com.xebialabs.deployit.engine.tasker.TaskExecutionEngine
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication
import com.xebialabs.deployit.repository.sql.CiRepository
import com.xebialabs.deployit.repository.{DeployedApplicationsRepository, WorkDir}
import com.xebialabs.deployit.security.Permissions
import com.xebialabs.deployit.security.client.PermissionService
import com.xebialabs.deployit.security.permission.{PermissionHelper, PlatformPermissions}
import com.xebialabs.deployit.spring.BeanWrapper
import com.xebialabs.overthere.local.LocalFile
import com.xebialabs.xlplatform.sugar.TempDirectorySugar
import grizzled.slf4j.Logging
import com.xebialabs.deployit.repository.sql.base.idToPath
import org.joda.time.DateTime
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import java.util.{UUID, List => JList}
import scala.jdk.CollectionConverters._

@Service
class ApplicationStatusServiceImpl @Autowired()(deployedApplicationsRepository: DeployedApplicationsRepository,
                                                ciRepository: CiRepository,
                                                engine: BeanWrapper[TaskExecutionEngine],
                                                permissionService: PermissionService,
                                                repositoryResource: RepositoryResource)

  extends ApplicationStatusService with TempDirectorySugar with Logging {

  val defaultOrderingField = "applicationName"
  val defaultOrderingDirection = "ASC"

  override def getApplicationStatus(deployedAppName: String, path: String, exactPath: Boolean = false, order: Ordering): JList[ApplicationDeploymentPackageState] = {
    val (orderingField, direction) = if (order == null) (defaultOrderingField, defaultOrderingDirection) else (order.field, order.direction.toString)

    val paths = deployedApplicationsRepository.find(deployedAppName, path, exactPath).map(_.getId).asJava
    val deployedApplications = withTempDirectory { tempDir =>
      val workDir = new WorkDir(LocalFile.from(tempDir.toFile))
      ciRepository.read[DeployedApplication](paths, workDir, 1,
        useCache = true,
        decryptPasswords = false,
        skipNotExistingCis = true).asScala
    }

    val allTasks = engine.get().getAllIncompleteTasks(FetchMode.SUMMARY).asScala.toVector

    val taskStates = allTasks.flatMap(task => {
      if (Option(task.getState).isDefined) {
        val taskType = TaskType.valueOf(DeploymentStatusUtils.extractMetadata(TaskMetadata.TASK_TYPE)(task))
        taskType match {
          case TaskType.CONTROL => None
          case _ =>
            val envDirRef = DeploymentStatusUtils.extractMetadata(TaskMetadata.ENVIRONMENT_DIRECTORY_REFERENCE)(task)
            val appDirRef = DeploymentStatusUtils.extractMetadata(TaskMetadata.APPLICATION_DIRECTORY_REFERENCE)(task)
            val satisfiesDeployedAppNameFilter = Option(deployedAppName) match {
              case Some(value) => DeploymentStatusUtils.extractMetadata(TaskMetadata.APPLICATION)(task).contains(value)
              case None => true
            }
            val satisfiesPathFilter = Option(path) match {
              case Some(value) =>
                val applicationPath = idToPath(DeploymentStatusUtils.getApplicationIdFromVersionId(DeploymentStatusUtils.extractMetadata(TaskMetadata.VERSION_ID)(task)))
                val filterPath = idToPath(value)
                if(exactPath) applicationPath.equals(filterPath)
                else applicationPath.startsWith(filterPath)
              case None => true
            }

            if (satisfiesDeployedAppNameFilter && satisfiesPathFilter && hasPermission(List(envDirRef, appDirRef))) {
              Option(DeploymentStatusUtils.extractMetadata(TaskMetadata.APPLICATION)(task) ->
                (DeploymentStatusUtils.extractMetadata(TaskMetadata.ENVIRONMENT)(task) ->
                  DeploymentDetails(
                    DeploymentStatusUtils.extractMetadata(TaskMetadata.VERSION)(task),
                    task.getState.toString,
                    DeploymentStatusUtils.extractMetadata(TaskMetadata.TASK_TYPE)(task),
                    task.getOwner,
                    Option(task.getStartDate).orElse(Option(task.getScheduledDate)).getOrElse(
                      DateTime.now()
                    ),
                    DeploymentStatusUtils.getApplicationPathFromTaskId(task.getId)
                  )
                  ))
            } else {
              None
            }
        }
      } else {
        None
      }
    }).groupBy(_._1).map {
      case (appName, values) =>
        appName -> values.map(_._2).toMap
    }

    val deployedAppStates = deployedApplications.map(app => app.getName ->
      (app.getEnvironment.getName ->
        DeploymentDetails(
          app.getVersion.getVersion,
          TaskExecutionState.DONE.toString,
          "",
          app.get$ciAttributes().getLastModifiedBy,
          app.get$ciAttributes().getLastModifiedAt,
          DeploymentStatusUtils.getApplicationPathFromVersionId(app.getVersion.getId)
        )
      )
    ).groupBy(_._1).map {
      case (appName, values) =>
        appName -> values.map(_._2).toMap
    }

    def reduceEnvs: (Map[String, DeploymentDetails], Map[String, DeploymentDetails]) => Map[String, DeploymentDetails] = {
      (done, ongoing) => {
        List(done, ongoing).flatten.groupMapReduce(_._1)(_._2)((_, o) => o)
      }
    }

    val applicationStates = List(deployedAppStates, taskStates).flatten.groupMapReduce(_._1)(_._2)(reduceEnvs)
      .flatMap {
        case (appName, details) =>
          details.map {
            case (envName, detail) =>
              ApplicationDeploymentPackageState(
                appName,
                detail.applicationPath,
                DeploymentPackageState(
                  envName,
                  "",
                  VersionTag(detail.version, ""),
                  detail.state,
                  detail.deploymentType,
                  detail.user,
                  detail.time)
              )
          }
      }.toList.sortBy(s => orderingField match {
        case "applicationName" => s.applicationName
        case "destination" => s.state.destination
        case "versionTag" => s.state.versionTag.label
        case "deploymentStatus" => s.state.deploymentStatus
        case "user" => s.state.user
        case "lastChangeTime" => s.state.lastChangeTime.getMillis.toString
        case _ => s.applicationName
    })

    val applicationStatesData = direction match {
      case "DESC" => applicationStates.reverse.asJava
      case _ => applicationStates.asJava
    }

    applicationStatesData
  }

  private def hasPermission(securedDirectoryReferences: List[String]): Boolean = {
    PermissionHelper.isCurrentUserAdmin || permissionService.checkPermission(
      securedDirectoryReferences.map(UUID.fromString).asJava, List(PlatformPermissions.READ).asJava, Permissions.getAuthentication)
  }
}

final case class DeploymentDetails(version: String, state: String, deploymentType: String, user: String, time: DateTime, applicationPath: String)
