package com.xebialabs.deployit.core.listener.version

import com.xebialabs.deployit.core.events.{CreateApplicationDeploymentPackage, DeleteApplicationDeploymentPackage}
import com.xebialabs.deployit.engine.api.dto.{ApplicationDeploymentPackage, VersionTag}
import com.xebialabs.deployit.engine.spi.event._
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.listener.version.readApplication
import com.xebialabs.deployit.plugin.api.udm.Version
import com.xebialabs.xldeploy.status.queue.DeploymentPackageStatusStoreHolder
import grizzled.slf4j.Logging
import nl.javadude.t2bus.Subscribe

import scala.jdk.CollectionConverters._

@DeployitEventListener
class VersionPropagationListener extends Logging {

  import com.xebialabs.deployit.core.util.IdExtensions._

  @Subscribe
  def receiveCisCreated(event: CisCreatedEvent): Unit = {
    event.getCis.asScala.foreach {
      case version: Version =>
        val application = readApplication(version)
        createApplicationDeploymentPackage(application.getName, version.getVersion)
      case _ =>
    }
  }

  @Subscribe
  def receiveCisCopied(event: CiCopiedEvent): Unit =
    event.getCi match {
      case version: Version =>
        val application = readApplication(version)
        val oldVersion = event.getEntity.getName
        val newVersion = event.getNewId.getName
        updateApplicationDeploymentPackage(application.getName, oldVersion, newVersion)
      case _ =>
    }

  @Subscribe
  def receiveCisDeleted(event: CisDeletedEvent): Unit =
    event.getCis.asScala.foreach {
      case version: Version =>
        val application = try {
          readApplication(version)
        } catch {
          case e: NotFoundException => event.getCis.asScala.find(ci => version.getId.startsWith(ci.getId)).getOrElse(throw e)
          case e: Throwable => throw e
        }
        deleteApplicationDeploymentPackage(application.getName, version.getVersion)
      case _ =>
    }

  @Subscribe
  def receiveCiRenamed(event: CiRenamedEvent): Unit =
    event.getCi match {
      case version: Version =>
        val application = readApplication(version)
        val oldVersion = event.getCi.getName
        val newVersion = event.getNewName
        updateApplicationDeploymentPackage(application.getName, oldVersion, newVersion)
      case _ =>
    }

  private def updateApplicationDeploymentPackage(applicationName: String, oldVersion: String, newVersion: String) = {
    deleteApplicationDeploymentPackage(applicationName, oldVersion)
    createApplicationDeploymentPackage(applicationName, newVersion)
  }

  private def createApplicationDeploymentPackage(applicationName: String, newVersion: String) = {
    DeploymentPackageStatusStoreHolder.getDeploymentPackageStatusStore.sendDeploymentPackageEvent(
      CreateApplicationDeploymentPackage(
        ApplicationDeploymentPackage(applicationName, VersionTag(newVersion, ""))
      )
    )
  }

  private def deleteApplicationDeploymentPackage(applicationName: String, oldVersion: String) =
    DeploymentPackageStatusStoreHolder.getDeploymentPackageStatusStore.sendDeploymentPackageEvent(
      DeleteApplicationDeploymentPackage(
        ApplicationDeploymentPackage(applicationName, VersionTag(oldVersion, ""))
      )
    )
}
