package com.xebialabs.deployit.core.rest.api;

import com.xebialabs.deployit.engine.api.dto.ConfigurationItemId;
import com.xebialabs.deployit.engine.api.dto.Deployment;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication;
import com.xebialabs.deployit.plugin.api.udm.EmbeddedDeployed;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class DeploymentWriter {
    private static final Logger logger = LoggerFactory.getLogger(DeploymentTaskServiceImpl.class);

    private DeploymentWriter() {
    }

    private static final Comparator<ConfigurationItemId> idComparator = Comparator.comparing(ConfigurationItemId::getId);

    private static List<ConfigurationItemId> cisToIds(Collection<? extends ConfigurationItem> cis) {
        return cis
                .stream()
                .map(DtoReader.ciToCiId::apply)
                .sorted(idComparator)
                .collect(Collectors.toList());
    }

    static Deployment createDeployment(DeployedApplication deployedApplication, Deployment.DeploymentType type) {
        Deployment deployment = new Deployment();
        deployment.setId(UUID.randomUUID().toString());
        deployment.setDeploymentType(type);
        deployment.setDeployedApplication(deployedApplication);

        logger.info("Create {} deployment {} for deployedApplication {}", type, deployment.getId(), deployedApplication.getId());

        List<List<Deployment>> requiredDeployments = new ArrayList<>();
        deployment.setGroupedRequiredDeployments(requiredDeployments);
        deployment.setDeployables(cisToIds(deployedApplication.getVersion().getDeployables()));
        deployment.setContainers(cisToIds(deployedApplication.getEnvironment().getMembers()));
        return deployment;
    }

    static List<ConfigurationItem> convertDeployeds(List<Deployed> initialDeployeds) {
        return Stream
                .concat(initialDeployeds.stream(), initialDeployeds.stream().flatMap(DeploymentWriter::getEmbeddeds))
                .collect(Collectors.toList());
    }

    private static Stream<ConfigurationItem> getEmbeddeds(final ConfigurationItem deployed) {
        Function<PropertyDescriptor, Stream<ConfigurationItem>> produceStream =
                (pd) -> ((Collection<ConfigurationItem>) pd.get(deployed)).stream();

        return deployed
                .getType()
                .getDescriptor()
                .getPropertyDescriptors()
                .stream()
                .filter(DeploymentWriter::isEmbeddedDeployedProperty)
                .flatMap(propertyDescriptor -> Stream.concat(
                        produceStream.apply(propertyDescriptor).flatMap(DeploymentWriter::getEmbeddeds),
                        produceStream.apply(propertyDescriptor)
                ));
    }

    private static boolean isEmbeddedDeployedProperty(PropertyDescriptor propertyDescriptor) {
        return propertyDescriptor.isAsContainment()
                && (propertyDescriptor.getKind() == PropertyKind.SET_OF_CI || propertyDescriptor.getKind() == PropertyKind.LIST_OF_CI)
                && propertyDescriptor.getReferencedType().isSubTypeOf(Type.valueOf(EmbeddedDeployed.class));
    }

}
