package com.xebialabs.deployit.core.rest.mappers;

import ai.digital.configuration.central.deploy.ClientProperties;
import com.thoughtworks.xstream.XStream;
import com.xebialabs.deployit.core.rest.resteasy.PathInterceptor;
import com.xebialabs.deployit.engine.spi.exception.DeployitException;
import com.xebialabs.deployit.engine.spi.exception.HttpResponseCodeResult;
import com.xebialabs.deployit.engine.spi.exception.SuppressStackTrace;
import com.xebialabs.xltype.serialization.xstream.XStreamReaderWriter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.ext.ExceptionMapper;
import javax.ws.rs.ext.Provider;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import static javax.ws.rs.core.HttpHeaders.ACCEPT;

// PLEASE NOTE: Do not remove the implemented interface, it is needed by RestEasy, even though the ClosingExceptionMapper also implements it.
@Provider
@Component
public class DeployitExceptionMapper extends ClosingExceptionMapper<DeployitException> implements ExceptionMapper<DeployitException> {

    private static final ThreadLocal<Boolean> doCDataWrapping = new ThreadLocal<>();

    private static final String logMsg = "Intercepting DeployitException";

    @Context
    private HttpServletRequest request;

    private final XStream xStream;

    public static void doCDataWrapping(boolean wrap) {
        doCDataWrapping.set(wrap);
    }

    @Autowired
    public DeployitExceptionMapper(ClientProperties config) {
        this(XStreamReaderWriter.getConfiguredXStream(), config);
    }

    public DeployitExceptionMapper(XStream xStream, ClientProperties config) {
        super(config.hideInternals());
        this.xStream = xStream;
    }

    @Override
    public Response handleException(DeployitException exception) {
        handleLogging(exception);

        int status = getResponseStatus(exception);

        String content = getContent(exception);

        Response resp = checkWhetherWeHideInternals(status, exception, content);

        if (resp != null) {
            return resp;
        }

        // Headers
        final Response.ResponseBuilder response = Response.status(status);
        response.header("X-Deployit-Exception", "true");
        response.header(X_EXCEPTION_TYPE, exception.getClass().getName());
        response.header(X_PATH, PathInterceptor.PATH.get());

        List<MediaType> acceptHeaders = determineAcceptMediaType();

        // Body: entity or error messages
        if (acceptHeaders.stream().anyMatch(h -> "binary".equals(h.getParameters().get("xl-exception")))) {
            response.entity(new ByteArrayInputStream(toBytes(exception)));
            response.type(MediaType.APPLICATION_OCTET_STREAM);
        } else {
            if (exception.hasEntity()) {
                if (acceptHeaders.stream().anyMatch(MediaType.APPLICATION_JSON_TYPE::isCompatible)) {
                    response.entity(DeployitExceptionHandler.getJsonEntity(exception));
                } else {
                    response.entity(content);
                    response.type(MediaType.APPLICATION_XML);
                }
                response.header("X-Entity", "true");
            } else {
                response.entity(content);
                response.type(MediaType.TEXT_PLAIN);
            }
        }

        return response.build();
    }

    private byte[] toBytes(final DeployitException exception) {
        try {
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            new ObjectOutputStream(out).writeObject(exception);
            return out.toByteArray();
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    protected List<MediaType> determineAcceptMediaType() {
        if (request.getHeader("Accept-Type") != null) {
            return Collections.singletonList(MediaType.valueOf(request.getHeader("Accept-Type")));
        } else if (request.getHeaders(ACCEPT) != null) {
            return Collections.list(request.getHeaders(ACCEPT)).stream()
                    .flatMap(h -> Arrays.stream(h.split(",")))
                    .map(val -> MediaType.valueOf(val.trim()))
                    .collect(Collectors.toList());
        } else {
            return Collections.singletonList(MediaType.APPLICATION_XML_TYPE);
        }
    }

    private String getContent(final DeployitException exception) {
        String content;
        if (exception.hasEntity()) {
            content = xStream.toXML(exception.getEntity());
        } else {
            content = Exceptions.getAllMessages(exception);
            Boolean wrap = doCDataWrapping.get();
            if (wrap != null && wrap) {
                content = "<![CDATA[" + content.replace("]]>", "___") + "]]>";
            }
        }
        return content;
    }

    private void handleLogging(DeployitException exception) {
        final SuppressStackTrace annotation = exception.getClass().getAnnotation(SuppressStackTrace.class);
        logger.info(exception.getMessage());
        if (annotation.suppressStackTrace()) {
            logger.trace(logMsg, exception);
        } else {
            logger.info(logMsg, exception);
        }
    }

    private int getResponseStatus(DeployitException exception) {
        final HttpResponseCodeResult annotation = exception.getClass().getAnnotation(HttpResponseCodeResult.class);
        int status = 500;
        if (annotation != null) {
            status = annotation.statusCode();
        }
        return status;
    }

}
