package com.xebialabs.deployit.core.rest.api;

import ai.digital.deploy.core.common.security.permission.DeployitPermissions;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.RemovalListener;
import com.google.common.cache.RemovalNotification;
import com.google.common.collect.FluentIterable;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.xebialabs.deployit.core.rest.resteasy.WorkdirHolder;
import com.xebialabs.deployit.engine.api.dto.AbstractDto;
import com.xebialabs.deployit.engine.api.dto.Deployment;
import com.xebialabs.deployit.engine.api.dto.SelectedDeployment;
import com.xebialabs.deployit.engine.api.execution.BlockState;
import com.xebialabs.deployit.engine.api.execution.StepState;
import com.xebialabs.deployit.engine.api.execution.TaskPreviewBlock;
import com.xebialabs.deployit.engine.spi.event.TaskCreatedEvent;
import com.xebialabs.deployit.engine.spi.event.TaskStartedEvent;
import com.xebialabs.deployit.engine.spi.exception.DeployitException;
import com.xebialabs.deployit.engine.tasker.*;
import com.xebialabs.deployit.event.DeployedChangeEventHandler;
import com.xebialabs.deployit.event.EventBusHolder;
import com.xebialabs.deployit.exception.NotFoundException;
import com.xebialabs.deployit.plugin.api.Deprecations;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.*;
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage;
import com.xebialabs.deployit.repository.DictionaryRepository;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.repository.WorkDir;
import com.xebialabs.deployit.server.api.util.IdGenerator;
import com.xebialabs.deployit.service.deployment.*;
import com.xebialabs.deployit.service.externalproperties.ExternalPropertiesResolver;
import com.xebialabs.deployit.service.validation.Validator;
import com.xebialabs.deployit.spring.BeanWrapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;

import javax.annotation.PreDestroy;
import java.util.*;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.google.common.base.Predicates.instanceOf;
import static com.google.common.base.Predicates.or;
import static com.google.common.collect.FluentIterable.from;
import static com.google.common.collect.Iterables.filter;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.uniqueIndex;
import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.checks.Checks.checkNotNull;
import static com.xebialabs.deployit.core.rest.api.DeploymentUtils.checkAndCast;
import static com.xebialabs.deployit.core.rest.api.DeploymentWriter.convertDeployeds;
import static com.xebialabs.deployit.core.rest.api.DeploymentWriter.createDeployment;
import static com.xebialabs.deployit.engine.api.dto.Deployment.DeploymentType.UNDEPLOYMENT;
import static java.lang.String.format;
import static java.util.concurrent.TimeUnit.MINUTES;
import static java.util.stream.Collectors.toCollection;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Stream.concat;

@Service
public class DeploymentTaskServiceImpl extends AbstractTaskResource implements DeploymentTaskService {

    @Autowired
    private RepositoryService repositoryService;

    @Autowired
    private ExternalPropertiesResolver externalPropertiesResolver;

    @Autowired
    private DeployedService deployedService;

    @Autowired
    private RollbackService rollbackService;

    @Autowired
    private BeanWrapper<TaskExecutionEngine> engine;

    @Autowired
    private Validator validator;

    @Autowired
    private PermissionChecker permissionChecker;

    @Autowired
    private DeploymentObjectGenerator deploymentObjectGenerator;

    @Autowired
    @Qualifier("taskSpecificationService")
    private TaskSpecificationService taskSpecificationService;

    @Autowired
    private DictionaryRepository dictionaryRepository;

    private final Cache<String, PreviewWithWorkdir<?>> previewCache = CacheBuilder.newBuilder().expireAfterAccess(5, MINUTES).removalListener(new PreviewCleaner()).build();

    @PreDestroy
    private void cleanCache() {
        previewCache.invalidateAll();
        previewCache.cleanUp();
    }

    @Override
    public boolean isDeployed(String applicationId, String environmentId) {
        logger.trace("isDeployed {}, {}", applicationId, environmentId);
        checkNotNull(applicationId, "application");
        checkNotNull(environmentId, "environment");
        checkCiExists(applicationId);
        checkCiExists(environmentId);
        return repositoryService.exists(IdGenerator.generateId(environmentId, applicationId));
    }

    @Override
    public Deployment prepareInitial(String versionId, String environmentId) {
        previewCache.cleanUp();

        logger.trace("prepareInitial {}, {}", versionId, environmentId);

        checkNotNull(versionId, "version");
        checkNotNull(environmentId, "environment");
        Version version = checkAndCast(repositoryService.read(versionId), Version.class);
        Environment env = checkAndCast(repositoryService.read(environmentId), Environment.class);
        externalPropertiesResolver.resolveExternalProperties(version, env);
        permissionChecker.checkPermission(DeployitPermissions.DEPLOY_INITIAL(), env);
        return deploymentObjectGenerator.forInitial(version, env);
    }

    @Override
    public Deployment prepareUpdate(String versionId, String deployedApplicationId) {
        previewCache.cleanUp();

        logger.trace("prepareUpgrade {}, {}", versionId, deployedApplicationId);
        checkNotNull(versionId, "version");
        checkNotNull(deployedApplicationId, "deployedApplication");

        DeployedApplication deployedApplication = checkAndCast(repositoryService.read(deployedApplicationId), DeployedApplication.class);
        Environment env = deployedApplication.getEnvironment();
        permissionChecker.checkPermission(DeployitPermissions.DEPLOY_UPGRADE(), env);
        Version newVersion = checkAndCast(repositoryService.read(versionId), Version.class);
        externalPropertiesResolver.resolveExternalProperties(deployedApplication, newVersion, env);

        return deploymentObjectGenerator.forUpdate(deployedApplication, newVersion, env);
    }

    @Override
    public Deployment prepareAutoDeployeds(Deployment deployment) {
        previewCache.cleanUp();

        logger.trace("updateDeployeds {}", deployment);
        DeployedApplication deployedApplication = preGenerateCheck(deployment);
        Version version = deployedApplication.getVersion();
        Environment environment = deployedApplication.getEnvironment();

        deployedService.scanMissingPlaceholders(deployedApplication, environment);
        List<Deployed> additionalDeployeds = deployedService.createDeployedsForNonExistingCombinations(deployment, version, environment, getDeployeds(deployment), new HashMap<>());
        permissionChecker.checkPermission(deployment, deployment.getGroupedRequiredDeployments());

        List<List<Deployment>> updatedRequiredDeployments = deployment.getGroupedRequiredDeployments().stream().map(s -> s.stream().map(this::prepareAutoDeployeds).collect(toList())).collect(toList());
        deployment.setGroupedRequiredDeployments(updatedRequiredDeployments);

        deployment.getDeployeds().clear();
        deployment.getResolvedPlaceholders().addAll(deployedApplication.get$ResolvedPlaceholders());
        deployment.addAll(convertDeployeds(additionalDeployeds));
        return deployment;
    }

    @Override
    public String createTask(Deployment deployment) {
        checkDeploymentInput(deployment);

        if (isNotValidDeploymentWithValidator(deployment)) {
            throw new InvalidDeploymentException(deployment);
        }

        invalidatePreview(deployment);

        logger.trace("Creating task for {}", deployment);
        warnAboutUsingDeprecatedCI(deployment);

        ResolvedPlaceholderGenerator.generateAndAdd(deployment);

        TaskSpecification deploymentTask = taskSpecificationService.createDeploymentTask(deployment, false);

        final String taskId = engine.get().register(deploymentTask);
        logger.debug("Registered {} task {} for Deployed Application {}", deployment.getDeploymentType(), taskId, deployment.getDeployedApplication().getId());

        EventBusHolder.publish(new TaskCreatedEvent(taskId, deployment.getDeploymentType().toString(), deployment.getDeployedApplication().getId()));
        notifyChangesInDeployeds(taskId, deployment);

        return taskId;
    }

    private void notifyChangesInDeployeds(String taskId, Deployment deployment) {
        Version version = deployment.getDeployedApplication().getProperty("version");
        Environment environment = deployment.getDeployedApplication().getProperty("environment");
        List<Deployed> existingDeployeds = ((DeployedApplication) deployment.getDeployedApplication())
                .getDeployeds()
                .stream()
                .flatMap(deployed ->
                        deployedService
                                .createSelectedDeployed(deployment, deployed.getDeployable(), deployed.getContainer(), deployed.getType(), version, environment)
                                .getDeployeds()
                                .stream()
                                .filter(existingDeployed -> existingDeployed.getId().equals(deployed.getId()))
                )
                .collect(Collectors.toList());

        DeployedChangeEventHandler.notifyDeployedPropertiesChange(taskId, deployment.getDeployeds(), convertDeployeds(existingDeployeds));
    }

    @Override
    public void startDeploymentTask(String taskId) {
        checkOwnership(taskId);
        taskQueueService.enqueueTask(taskId);
        EventBusHolder.publish(new TaskStartedEvent(taskId));
    }

    @Override
    public Deployment prepareUndeploy(String deployedApplicationId) {
        previewCache.cleanUp();
        logger.trace("prepareUndeployApplication {}", deployedApplicationId);
        checkNotNull(deployedApplicationId, "deployedApplication");
        DeployedApplication deployedApplication = checkAndCast(repositoryService.read(deployedApplicationId, 2), DeployedApplication.class);
        checkPermission(DeployitPermissions.UNDEPLOY(), deployedApplication.getEnvironment().getId());
        Deployment unDeployment = createDeployment(deployedApplication, UNDEPLOYMENT);
        taskSpecificationService.applyUnDeploymentGroups(deployedApplication, unDeployment);
        return unDeployment;
    }

    @Override
    public Deployment generateSelectedDeployeds(SelectedDeployment selectedDeployment) {
        previewCache.cleanUp();

        List<String> deployableIds = selectedDeployment.getSelectedDeployableIds();
        checkArgument(!deployableIds.isEmpty(), "Should select at least one deployable to generate a deployed");

        Deployment deployment = selectedDeployment.getDeployment();
        logger.trace("generateSelectedDeployeds {}, {}", deployableIds, deployment);
        DeployedApplication deployedApplication = preGenerateCheck(deployment);
        Version version = deployedApplication.getVersion();
        logger.debug("Generating deployeds from package [{}]", version);

        Set<Deployable> allDeployables = deployment.getRequiredDeployments().stream()
                .map(this::preGenerateCheck)
                .map(DeployedApplication::getVersion)
                .flatMap(dependencyVersion -> dependencyVersion.getDeployables().stream())
                .collect(toCollection(() -> new HashSet<>(version.getDeployables())));

        Map<String, Deployable> deployables = uniqueIndex(allDeployables, ConfigurationItem::getId);
        final List<ConfigurationItem> deployableCIs = deployableIds.stream()
                .peek(id -> checkArgument(deployables.containsKey(id), "All sources should be from same package or its dependencies."))
                .map(deployables::get)
                .collect(toList());

        Environment environment = deployedApplication.getEnvironment();
        deployedService.scanMissingPlaceholders(deployedApplication, environment);
        GeneratedDeployeds selectedDeployeds = deployedService.generateSelectedDeployeds(deployment, deployableCIs, version, environment);
        deployment.addAll(convertDeployeds(selectedDeployeds.getDeployeds()));
        return deployment;
    }

    @Override
    public Deployment generateSingleDeployed(String deployableId, String containerId, Type deployedType, Deployment deployment) {
        previewCache.cleanUp();

        logger.debug("Creating explicit deployed for [{}] and [{}]", deployableId, containerId);
        DeployedApplication deployedApplication = preGenerateCheck(deployment);
        Version version = deployedApplication.getVersion();
        Environment environment = deployedApplication.getEnvironment();

        checkNotNull(deployableId, "deployable");
        checkNotNull(containerId, "container");
        Deployable deployable = checkAndCast(Iterables.find(collectDeployables(deployment), input -> input.getId().equals(deployableId)), Deployable.class);
        Container container = checkAndCast(Iterables.find(environment.getMembers(), input -> input.getId().equals(containerId)), Container.class);

        GeneratedDeployeds selectedDeployed = deployedService.createSelectedDeployed(deployment, deployable, container, deployedType, version, environment);

        deployment.addAll(convertDeployeds(selectedDeployed.getDeployeds()));
        return deployment;
    }

    @Override
    public TaskPreviewBlock taskPreviewBlock(Deployment deployment) {
        checkDeploymentInput(deployment);

        try {
            invalidatePreview(deployment);
            PreviewWithWorkdir<TaskPreviewBlock> preview = getTaskPreview(deployment, WorkdirHolder.get(), new TaskPreviewBlockBuilder());
            return preview.getPreview();
        } catch (Exception e) {
            throw new DeployitException(e, "Error getting task preview for deployment [%s]", deployment.getId());
        }
    }

    @Override
    public StepState taskPreviewBlock(Deployment deployment, String blockId, int stepNr) {
        checkDeploymentInput(deployment);

        try {
            invalidatePreview(deployment);
            PreviewWithWorkdir<TaskPreviewBlock> preview = getTaskPreview(deployment, WorkdirHolder.get(), new TaskPreviewBlockBuilder());

            Block topBlock = (Block) preview.getPreview().getBlock();
            BlockState child = topBlock.getBlock(BlockPath.apply(blockId).tail()).getOrElse(null);
            if (!(child instanceof StepBlock)) {
                throw new DeployitException("Block [%s] is a composite block so it has no steps", blockId);
            }

            List<StepState> steps = ((StepBlock) child).getSteps();
            checkArgument(stepNr > 0 && stepNr <= steps.size(), "Not a valid step number [%s]", stepNr);
            StepState stepState = steps.get(stepNr - 1);
            if (hasPermission(DeployitPermissions.TASK_PREVIEWSTEP())) {
                return new PreviewStepAdapter((TaskStep) stepState);
            } else {
                return stepState;
            }
        } catch (Exception e) {
            throw new DeployitException(e, "Error getting task preview for deployment [%s]", deployment.getId());
        }
    }

    @Override
    public String rollback(String taskid) {
        Task task = engine.get().retrieveTask(taskid);
        return rollbackService.rollback(task);
    }

    @Override
    public Map<String, String> effectiveDictionary(String environment, String applicationVersion, String application, String container) {
        return dictionaryRepository.getDictionariesForEnv(environment, application, container, applicationVersion);
    }

    private DeployedApplication preGenerateCheck(Deployment deployment) {
        checkNotNull(deployment, "deployment");
        invalidatePreview(deployment);
        return (DeployedApplication) deployment.getDeployedApplication();
    }

    private void invalidatePreview(Deployment deployment) {
        previewCache.cleanUp();

        if (deployment.getId() != null) {
            previewCache.invalidate(deployment.getId());
        }
    }

    /**
     * Collects all deployables from deployment object
     * This method takes into account deployables from main application and dependencies
     *
     * @param deployment - deployment instance
     * @return collection of all found deployables
     */
    private List<Deployable> collectDeployables(Deployment deployment) {
        Stream<Deployable> dependantDeployables = deployment.getRequiredDeployments().stream().flatMap(depl -> ((DeployedApplication) depl.getDeployedApplication()).getVersion().getDeployables().stream());
        return concat(((DeployedApplication) deployment.getDeployedApplication()).getVersion().getDeployables().stream(), dependantDeployables).collect(toList());
    }

    @Override
    public Deployment validate(Deployment deployment) {
        previewCache.cleanUp();

        invalidatePreview(deployment);
        if (isNotValidDeploymentWithValidator(deployment)) {
            throw new InvalidDeploymentException(deployment);
        }

        return deployment;
    }

    private <T extends AbstractDto> PreviewWithWorkdir<T> getTaskPreview(final Deployment deployment, final WorkDir workDir, final PreviewBuilder<T> builder) throws ExecutionException {
        String id = deployment.getId();
        checkArgument(id != null, "Old-style client detected, deployment does not have an 'id' set. Please upgrade your client.");

        Callable<? extends PreviewWithWorkdir<?>> preview = () -> {
            logger.info("Generating {} for deployment [{}]", builder, deployment.getId());

            ConfigurationItem configurationItem = validateDeployedApplication(deployment.getDeployedApplication());
            if (!configurationItem.get$validationMessages().isEmpty()) {
                throw new InvalidDeploymentException("The DeployedApplication contained validation errors, not generating a steps view.");
            }
            filterDeployedsWithValidationErrors(deployment);
            TaskSpecification taskSpecification = taskSpecificationService.createDeploymentTask(deployment, true);

            logger.info("Done generating {} for deployment [{}]", builder, deployment.getId());

            return builder.build(deployment.getId(), workDir, taskSpecification);
        };
        @SuppressWarnings("unchecked")
        PreviewWithWorkdir<T> possiblyCached = (PreviewWithWorkdir<T>) previewCache.get(id, preview);

        return possiblyCached;
    }

    private void filterDeployedsWithValidationErrors(Deployment deployment) {
        for (ConfigurationItem configurationItem : newArrayList(deployment.getDeployeds())) {
            if (!validator.validate(configurationItem).isEmpty()) {
                logger.debug("ConfigurationItem [{} ({})] contained validation errors, not taking it into account for steps view.", configurationItem.getId(), configurationItem.getType());
                deployment.getDeployeds().remove(configurationItem);
            }
        }
    }

    private void warnAboutUsingDeprecatedCI(final Deployment deployment) {
        DeployedApplication da = ((DeployedApplication) deployment.getDeployedApplication());

        if (Type.valueOf(CompositePackage.class).equals(da.getVersion().getType())) {
            Deprecations.deprecated("Deprecation warning: You are deploying a udm.CompositePackage [{}]. " +
                    "Please use application dependencies instead. Refer to the upgrade manual for more information.", da.getVersion().getId());
        }
    }

    private void checkDeploymentInput(Deployment deployment) {
        checkNotNull(deployment, "deployment");
        permissionChecker.checkPermission(deployment, deployment.getGroupedRequiredDeployments());
        checkDeployedDuplicates(deployment);
    }

    /**
     * Make sure that there are no duplicate configuration items in provided deployeds lists
     *
     * @param deployment - deployment object
     * @throws com.xebialabs.deployit.core.rest.api.InvalidDeploymentException - in case when duplicate deployeds found
     */
    private void checkDeployedDuplicates(Deployment deployment) {
        Iterable<ConfigurationItem> deployeds = FluentIterable.from(deployment.getRequiredDeployments()).transformAndConcat(input -> input.getDeployeds()).append(deployment.getDeployeds());
        Set<String> processed = new HashSet<>();
        for (ConfigurationItem deployed : deployeds) {
            if (!processed.contains(deployed.getId())) {
                processed.add(deployed.getId());
            } else {
                throw new InvalidDeploymentException("You are trying to deploy multiple items with the same id: " + deployed.getId() + " on the same container.");
            }
        }
    }

    @SuppressWarnings("rawtypes")
    private static Set<Deployed> getDeployeds(Deployment deployment) {
        Collection<ConfigurationItem> deployedEntities = deployment.getDeployeds();
        FluentIterable<ConfigurationItem> from = from(deployedEntities);
        checkArgument(from.allMatch(or(instanceOf(Deployed.class), instanceOf(EmbeddedDeployed.class))), "The Deployment can only contain Deployed or EmbeddedDeployed configuration items");
        return Sets.newHashSet(from.filter(Deployed.class));
    }

    private boolean isNotValidDeploymentWithValidator(Deployment deployment) {
        ConfigurationItem deployedApplication = validateDeployedApplication(deployment.getDeployedApplication());
        deployment.setDeployedApplication(deployedApplication);
        boolean deployedApplicationValid = deployedApplication.get$validationMessages().isEmpty();
        boolean deployedsValid = true;
        try {
            List<ConfigurationItem> cis = newArrayList(deployment.getDeployeds());
            for (Deployment d : deployment.getRequiredDeployments()) {
                cis.addAll(d.getDeployeds());
            }
            validator.validateCis(cis);
        } catch (Validator.ValidationsFailedException ignored) {
            deployedsValid = false;
        }

        deployedsValid = deployedsValid && updateWithValidatedDeployeds(deployment);

        for (Deployment depl : deployment.getRequiredDeployments()) {
            deployedsValid = deployedsValid && updateWithValidatedDeployeds(depl);
        }

        return !(deployedsValid && deployedApplicationValid);
    }

    private boolean updateWithValidatedDeployeds(Deployment deployment) {
        List<ConfigurationItem> validatedDeployeds = newArrayList();
        boolean deployedsValid = validateDeployeds(deployment.getDeployeds(), validatedDeployeds);
        deployment.setDeployeds(validatedDeployeds);
        return deployedsValid;
    }

    private boolean validateDeployeds(List<ConfigurationItem> deployeds, List<ConfigurationItem> validated) {
        boolean deployedsValid = true;
        Set<String> ids = newHashSet();

        for (ConfigurationItem configurationItem : deployeds) {
            if (!ids.add(configurationItem.getId())) {
                ValidationMessage msg = new ValidationMessage(configurationItem.getId(), "name", format("The deployed [%s] must have a unique name within the container [%s]",
                        configurationItem, ((EmbeddedDeployedContainer<?, ?>) configurationItem).getContainer()));
                configurationItem.get$validationMessages().add(msg);
                validated.add(configurationItem);
                deployedsValid = false;
            } else {
                validated.add(configurationItem);
            }
        }
        return deployedsValid;
    }

    @SuppressWarnings("unchecked")
    private ConfigurationItem validateDeployedApplication(ConfigurationItem deployedApplicationEntity) {
        List<ValidationMessage> messages = validator.validate(deployedApplicationEntity, Collections.emptyList());
        //filter out any deployed property error messages on the deployed application
        Iterable<ValidationMessage> filteredMessages = filter(messages, input -> !"deployeds".equals(input.getPropertyName()));
        if (Iterables.size(filteredMessages) > 0) {
            deployedApplicationEntity.get$validationMessages().addAll(Lists.newArrayList(filteredMessages));
        }
        return deployedApplicationEntity;
    }

    private void checkCiExists(String ciId) {
        if (!repositoryService.exists(ciId)) {
            throw new NotFoundException("Repository entity [%s] not found", ciId);
        }
    }

    private static class PreviewWithWorkdir<T extends AbstractDto> {
        private final T preview;
        private final WorkDir workDir;

        private PreviewWithWorkdir(T preview, WorkDir workDir) {
            this.preview = preview;
            this.workDir = workDir;
        }

        void cleanUp() {
            workDir.delete();
        }

        public T getPreview() {
            return preview;
        }

        @Override
        protected void finalize() throws Throwable {
            super.finalize();
            workDir.delete();
        }
    }

    private static class PreviewCleaner implements RemovalListener<String, PreviewWithWorkdir<?>> {
        @Override
        public void onRemoval(RemovalNotification<String, PreviewWithWorkdir<?>> notification) {
            if (notification.getValue() != null) {
                logger.info("Cleaning up Preview [{}] with workDir [{}]", notification.getKey(), notification.getValue().workDir);
                notification.getValue().cleanUp();
            }
        }
    }

    private interface PreviewBuilder<T extends AbstractDto> {
        PreviewWithWorkdir<T> build(String deploymentId, WorkDir workDir, TaskSpecification taskSpecification);
    }

    private static final class TaskPreviewBlockBuilder implements PreviewBuilder<TaskPreviewBlock> {
        @Override
        public PreviewWithWorkdir<TaskPreviewBlock> build(String deploymentId, WorkDir workDir, TaskSpecification taskSpecification) {
            TaskPreviewBlock preview = new TaskPreviewBlock(deploymentId, taskSpecification.getBlock());
            return new PreviewWithWorkdir<>(preview, workDir);
        }

        @Override
        public String toString() {
            return "task preview block";
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(DeploymentTaskServiceImpl.class);
}
