package com.xebialabs.deployit.core.rest.api.reports;

import ai.digital.deploy.sql.model.Report;
import ai.digital.deploy.sql.model.ReportLine;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.xebialabs.deployit.task.archive.TaskArchive;

import java.util.Map;

import static com.xebialabs.deployit.core.util.CiUtils.getIdWithoutRoot;
import static com.xebialabs.deployit.core.util.CiUtils.getName;
import static com.xebialabs.deployit.task.archive.sql.schema.ArchivedDeploymentTasks.environment;
import static com.xebialabs.deployit.task.archive.sql.schema.ArchivedDeploymentTasks.main_application;

public class DeploymentsKeyIndicator extends TaskArchiveReport {

    public DeploymentsKeyIndicator(TaskArchive taskArchive) {
        super(taskArchive);
    }

    public Report report() {
        final LoadingCache<String, TaskStateData> tasks = CacheBuilder.newBuilder().build(CacheLoader.from(input -> new TaskStateData()));

        taskArchive.searchForMaps(getSearchParameters(), true).forEach(task -> addTask(tasks, task));

        return generateReport(tasks.asMap());
    }

    private static Report generateReport(Map<String, TaskStateData> tasksMap) {
        final Report report = new Report();
        for (Map.Entry<String, TaskStateData> entry : tasksMap.entrySet()) {
            TaskStateData task = entry.getValue();
            final ReportLine line = report.addLine();
            String[] appAndEnv = entry.getKey().split("~", 2);
            String appId = appAndEnv[0];
            String envId = appAndEnv[1];
            line.addValue("application", appId);
            line.addValue("environment", getName(envId));
            line.addValue("environmentId", envId);
            line.addValue("environmentIdWithoutRoot", getIdWithoutRoot(envId));
            line.addValue("noOfSuccessfulDeployments", task.getNoOfSuccessfulDeployments().toString());
            line.addValue("noOfFailedDeployments", task.getNoOfFailedDeployments().toString());
            line.addValue("noOfAbortedDeployments", task.getNoOfAbortedDeployments().toString());
            line.addValue("noOfRollbacks", task.getNoOfRollbacks().toString());
            line.addValue("averageDuration", getAverageDeploymentTime(task));
        }
        return report;
    }

    static String getAverageDeploymentTime(TaskStateData task) {
        if (task.getTotalSuccessfulTasks() > 0) {
            return ReportUtils.formatToHoursMinsSecs(task.getAverageDeploymentTime() / task.getTotalSuccessfulTasks());
        }
        return "00:00:00";
    }

    private static void addTask(LoadingCache<String, TaskStateData> tasks, Map<String, Object> task) {
        String entityId = task.get(main_application().name()) + "~" + task.get(environment().name());
        tasks.getUnchecked(entityId).addTaskAndDuration(task);
    }
}
