package com.xebialabs.deployit.core.rest.api;

import com.xebialabs.deployit.core.rest.resteasy.Workdir;
import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource;
import com.xebialabs.deployit.engine.api.PackageService;
import com.xebialabs.deployit.engine.api.dto.FileUpload;
import com.xebialabs.deployit.engine.api.dto.ImportFromUrlParams;
import com.xebialabs.deployit.engine.spi.event.PackageImportedEvent;
import com.xebialabs.deployit.event.EventBusHolder;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.credentials.UsernamePasswordCredentials;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.repository.WorkDirFactory;
import com.xebialabs.deployit.server.api.importer.ImportSource;
import com.xebialabs.deployit.service.importer.ImporterException;
import com.xebialabs.deployit.service.importer.ImporterService;
import com.xebialabs.deployit.service.importer.source.FileSource;
import com.xebialabs.deployit.service.importer.source.InputStreamSource;
import com.xebialabs.deployit.service.importer.source.UrlSource;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import scala.Tuple2;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.List;

import static java.lang.String.format;


/**
 * I am A REST resource that exposes functionality on the Deployable packages.
 * <p/>
 * Functionality includes: - Listing all Deployment packages in the importablePackages directory on the server.
 */
@Service
public class PackageResource extends AbstractSecuredResource implements PackageService {

    @Autowired
    private ImporterService importerService;

    @Autowired
    private RepositoryService repositoryService;

    @Override
    public List<String> list() {
        return importerService.listPackages();
    }

    @Override
    @Workdir(prefix = WorkDirFactory.IMPORT_WORKDIR_PREFIX)
    public ConfigurationItem importPackage(String file) {
        if (!importerService.getImportablePackageDirectory().exists()) {
            throw new ImporterException(format("Directory '%s' to import packages from does not exist", importerService.getImportablePackageDirectory().getAbsolutePath()));
        }
        File packageFile = new File(importerService.getImportablePackageDirectory(), file);
        if (!packageFile.exists()) {
            throw new ImporterException(format("Package '%s' does not exist", packageFile.getAbsolutePath()));
        }
        FileSource fileSource = new FileSource(packageFile, false);
        return doImport(fileSource);
    }

    @Override
    @Workdir(prefix = WorkDirFactory.IMPORT_WORKDIR_PREFIX)
    public ConfigurationItem fetch(String url) {
        try {
            UrlSource urlSource = new UrlSource(new URL(url), null, null);
            return doImport(urlSource);
        } catch (MalformedURLException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    @Workdir(prefix = WorkDirFactory.IMPORT_WORKDIR_PREFIX)
    public ConfigurationItem fetch(ImportFromUrlParams params) {
        try {
            Tuple2<String, String> credentials = getCredentials(params);

            UrlSource urlSource = new UrlSource(new URL(params.getUrl()), credentials._1, credentials._2);
            return doImport(urlSource);
        } catch (IllegalArgumentException | MalformedURLException e) {
            throw new RuntimeException(e);
        }
    }

    private Tuple2<String, String> getCredentials(ImportFromUrlParams params) throws IllegalArgumentException {

        if (!StringUtils.isBlank(params.getUser()) && !StringUtils.isBlank(params.getCredentialsId())) {
            throw new IllegalArgumentException("Can not use both style of credentials");
        }
        if (!StringUtils.isBlank(params.getUser())) {
            return new Tuple2<>(params.getUser(), params.getPassword());
        }
        if (!StringUtils.isBlank(params.getCredentialsId())) {
            UsernamePasswordCredentials credentials = repositoryService.read(params.getCredentialsId());
            return new Tuple2<>(credentials.getUsername(), credentials.getPassword());
        }
        return new Tuple2<>(null, null);
    }

    @Override
    @Workdir(prefix = WorkDirFactory.IMPORT_WORKDIR_PREFIX)
    public ConfigurationItem upload(String file, FileUpload form) {
        try (InputStream packageStream = new FileInputStream(form.getFileData())) {
            InputStreamSource byteArraySource = new InputStreamSource(file, packageStream);
            return doImport(byteArraySource);
        } catch (IOException e) {
            throw new IllegalStateException(format("Error uploading file: %s", file));
        }
    }

    private ConfigurationItem doImport(ImportSource source) {
        String importedApplication = importerService.importPackage(source);
        ConfigurationItem read = repositoryService.read(importedApplication);

        EventBusHolder.publish(new PackageImportedEvent(importedApplication));

        return read;
    }
}
