package com.xebialabs.xldeploy.auth.config

import com.xebialabs.deployit.booter.local.utils.Strings
import com.xebialabs.deployit.core.util.TokenGenerator
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.security.{PermissionEnforcer, UserService}
import com.xebialabs.deployit.security.authentication.{AuthenticationFailureException, PersonalAuthenticationToken}
import com.xebialabs.deployit.security.principaldata.PrincipalDataProvider
import com.xebialabs.deployit.security.service.{UserGroupService, UserTokenService}
import com.xebialabs.xldeploy.auth.TokenExpiredException
import org.slf4j.LoggerFactory
import org.springframework.security.authentication.{AuthenticationProvider, BadCredentialsException, UsernamePasswordAuthenticationToken}
import org.springframework.security.core.authority.SimpleGrantedAuthority
import org.springframework.security.core.{Authentication, GrantedAuthority}
import org.springframework.security.core.authority.mapping.{GrantedAuthoritiesMapper, NullAuthoritiesMapper}
import org.springframework.stereotype.Component

import java.text.SimpleDateFormat
import java.util
import java.util.Collections

@Component("xlAuthenticationProvider")
class DeployAuthenticationProvider(userService: UserService,
                                   userTokenService: UserTokenService,
                                   principalDataProvider: PrincipalDataProvider ,
                                   userGroupService:  UserGroupService)extends AuthenticationProvider{
  private val logger = LoggerFactory.getLogger(classOf[DeployAuthenticationProvider])
  private var authoritiesMapper: GrantedAuthoritiesMapper = new NullAuthoritiesMapper()

  override def authenticate(token: Authentication): Authentication = {
    logger.debug("Authenticating for Digital.ai Deploy")
    try {
      token match {
        case _: PersonalAuthenticationToken =>
          authenticateFromUserToken(token.getCredentials.toString)
        case _ =>
          authenticateFromCredentials(token.getPrincipal.toString, token.getCredentials.toString)
      }
    } catch {
      case exc@(_: AuthenticationFailureException | _: NotFoundException) =>
        throw new BadCredentialsException(exc.getMessage, exc)
    }
  }

  private def authenticateFromUserToken(token: String): Authentication = {
    logger.trace("Authenticating using personal access token")
    val tokenHash = TokenGenerator.hash(token)
    val userToken = userTokenService.findByUserToken(tokenHash)
    if(userToken == null) throw new AuthenticationFailureException("Cannot authenticate with supplied personal access token")

    if (userToken.tokens.head.isExpired()) {
      throw TokenExpiredException(s"The token expired on ${new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssXXX").format(userToken.tokens.head.expiryDate)}")
    }

    userTokenService.updateLastUsedDate(userToken.tokens.head.ciId, new java.util.Date())
    val mappedAuthorities: util.Collection[_ <: GrantedAuthority] = evaluateAuthorities(userToken.username)
    new PersonalAuthenticationToken(userToken.username,
      token,
      userToken.tokens.head.expiryDate,
      mappedAuthorities,
      Collections.emptySet(),
      Collections.emptySet(),
      Collections.emptySet())
  }

  private def authenticateFromCredentials(username: String, password: String): Authentication = {
    logger.trace(s"Authenticating [$username]")
    if (Strings.isBlank(username)) {
      throw new BadCredentialsException("Cannot authenticate with empty username")
    }
    userService.authenticate(username, password)
    val mappedAuthorities: util.Collection[_ <: GrantedAuthority] = evaluateAuthorities(username)
    new UsernamePasswordAuthenticationToken(username, password, mappedAuthorities)
  }

  override def supports(authentication: Class[_]): Boolean = {
    authentication.isAssignableFrom(classOf[UsernamePasswordAuthenticationToken]) ||
      authentication.isAssignableFrom(classOf[PersonalAuthenticationToken])
  }

  private def evaluateAuthorities(username: String): util.Collection[_ <: GrantedAuthority] = {
    val userData = principalDataProvider.getUserData(username)
    if (userData.isFound) {
      authoritiesMapper.mapAuthorities(principalDataProvider.getAuthorities(username))
    } else {
      val authorities = try {
        val user = userService.read(username)
        val grantedAuthorities = new util.ArrayList[GrantedAuthority]
        if (user.isAdmin || "admin" == user.getUsername) {
          grantedAuthorities.add(new SimpleGrantedAuthority(PermissionEnforcer.ROLE_ADMIN))
        }
        grantedAuthorities
      } catch {
        case _: NotFoundException =>
          // map authorities for SSO users like OIDC from saved group membership
          val userGroups = userGroupService.findGroupsForUser(username)
          val grantedAuthorities = new util.ArrayList[GrantedAuthority]
          userGroups.foreach { group =>
            grantedAuthorities.add(new SimpleGrantedAuthority(group))
          }
          grantedAuthorities
      }
      authoritiesMapper.mapAuthorities(authorities)
    }
  }


  def getAuthoritiesMapper: GrantedAuthoritiesMapper = authoritiesMapper

  def setAuthoritiesMapper(authoritiesMapper: GrantedAuthoritiesMapper): Unit = {
    this.authoritiesMapper = authoritiesMapper
  }
}
