package com.xebialabs.xldeploy.auth.config

import ai.digital.config.ServerConfigurationHelper
import ai.digital.configuration.central.deploy.{ClientProperties, ServerSideProperties}
import com.xebialabs.deployit.core.auth.LoginMetadataService
import com.xebialabs.deployit.engine.api.distribution.TaskExecutionWorkerRepository
import com.xebialabs.deployit.plumbing.authentication.WithoutRedirectLoginSuccessHandler
import com.xebialabs.deployit.security.RoleService
import com.xebialabs.deployit.security.authentication.{BasicAuthWithRememberMeFilter, RememberMeAuthenticationProvider, XLCrowdAuthenticationProvider}
import com.xebialabs.deployit.taskexecution.security.{TaskWorkerAuthenticationFilter, TaskWorkerAuthenticationProvider}
import com.xebialabs.deployit.{LicenseExpiryCheckFilter, LogbackAccessSecurityAttributesSaveFilter}
import com.xebialabs.license.LicenseValidationFilter
import com.xebialabs.license.service.LicenseService
import com.xebialabs.xldeploy.auth.{BasicAuthOverridingHttpSessionSecurityContextRepository, XlPersistentTokenBasedRememberMeServices}
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.{Autowired, Qualifier, Value}
import org.springframework.context.ApplicationContext
import org.springframework.context.annotation.Lazy
import org.springframework.http.HttpMethod
import org.springframework.security.access.AccessDecisionManager
import org.springframework.security.authentication.AuthenticationProvider
import org.springframework.security.config.annotation.authentication.builders.AuthenticationManagerBuilder
import org.springframework.security.config.annotation.web.builders.{HttpSecurity, WebSecurity}
import org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter
import org.springframework.security.config.http.SessionCreationPolicy
import org.springframework.security.ldap.authentication.LdapAuthenticationProvider
import org.springframework.security.web.access.AccessDeniedHandler
import org.springframework.security.web.authentication.logout.HttpStatusReturningLogoutSuccessHandler
import org.springframework.security.web.authentication.{AuthenticationFailureHandler, DelegatingAuthenticationEntryPoint, UsernamePasswordAuthenticationFilter}
import org.springframework.security.web.context.request.async.WebAsyncManagerIntegrationFilter
import org.springframework.security.web.firewall.StrictHttpFirewall
import org.springframework.security.web.savedrequest.NullRequestCache

import scala.jdk.CollectionConverters.ListHasAsScala

abstract class DeploySecurityConfig extends WebSecurityConfigurerAdapter {

  private val logger = LoggerFactory.getLogger(classOf[DeploySecurityConfig])

  @Value("${deploy.server.license.days-before-warning:5}")
  var daysBeforeWarning: Int = _

  @Autowired
  var applicationContext: ApplicationContext = _

  @Autowired
  var loginMetadataService: LoginMetadataService = _

  @Autowired
  var roleService: RoleService = _

  @Autowired
  var accessDeniedHandler: AccessDeniedHandler = _

  @Autowired
  @Qualifier("rememberMeKey")
  var rememberMeKey: String = _

  @Autowired
  var rememberMeServices: XlPersistentTokenBasedRememberMeServices = _

  @Autowired
  var withoutRedirectLogoutSuccessHandler: HttpStatusReturningLogoutSuccessHandler = _

  @Autowired
  var workerRepository: TaskExecutionWorkerRepository = _

  @Autowired
  var licenseService: LicenseService = _

  @Autowired
  var taskWorkerAuthenticationProvider: TaskWorkerAuthenticationProvider = _

  @Autowired(required = false)
  var ldapAuthenticationProvider: java.util.List[LdapAuthenticationProvider] = _

  @Autowired(required = false)
  var xlCrowdAuthenticationProvider: java.util.List[XLCrowdAuthenticationProvider] = _

  @Autowired
  var accessDecisionManager: AccessDecisionManager = _

  @Lazy
  @Autowired
  @Qualifier("xlAuthenticationProvider")
  var xlAuthenticationProvider: AuthenticationProvider = _

  @Autowired
  var delegatingAuthenticationEntryPoint: DelegatingAuthenticationEntryPoint = _

  @Autowired
  var delegatingSecurityContextRepository: BasicAuthOverridingHttpSessionSecurityContextRepository = _

  @Autowired
  var authenticationFailureHandler: AuthenticationFailureHandler = _

  @Autowired
  var serverSideConfiguration: ServerSideProperties = _

  @Autowired
  var clientProperties: ClientProperties = _

  override def configure(auth: AuthenticationManagerBuilder): Unit = {

    InMemoryConfigurer.configure(auth)

    auth
      .authenticationProvider(taskWorkerAuthenticationProvider)
      .authenticationProvider(new RememberMeAuthenticationProvider())
      .authenticationProvider(xlAuthenticationProvider)

    if (ldapAuthenticationProvider != null) {
      for(provider <- ldapAuthenticationProvider.asScala) {
        auth
          .authenticationProvider(provider)
      }
    }

    if (xlCrowdAuthenticationProvider != null) {
      for(provider <- xlCrowdAuthenticationProvider.asScala) {
        auth
          .authenticationProvider(provider)
      }
    }
  }

  def configureSecurity(web: WebSecurity, prefix: String): Unit = {
    val firewall = new StrictHttpFirewall()
    firewall.setAllowUrlEncodedSlash(true)
    firewall.setAllowUrlEncodedPercent(true)
    web.httpFirewall(firewall)

    web.ignoring().antMatchers("/productregistration")
    web.ignoring().antMatchers(s"/$prefix/internal/configuration/properties")
    web.ignoring().antMatchers(HttpMethod.GET, s"/$prefix/settings/general")
    web.ignoring().antMatchers(s"/$prefix/ha/health")
    if (clientProperties.ignoreAuthCheckForServerStateApi)
      web.ignoring().antMatchers(s"/$prefix/server/state")

    web.ignoring().antMatchers("/**.js**")
    web.ignoring().antMatchers("/**.html")
    web.ignoring().antMatchers("/**.css")
    web.ignoring().antMatchers("/favicon.ico")
    web.ignoring().antMatchers("/fonts/**")
    web.ignoring().antMatchers("/icons/**")
    web.ignoring().antMatchers("/images/**")
  }

  def licenseValidationFilter: LicenseValidationFilter = {
    val filter = new LicenseValidationFilter()
    filter.setLicenseService(licenseService)
    filter
  }

  def configureSecurity(http: HttpSecurity, prefix: String): Unit = {
    http
      .securityContext()
      .securityContextRepository(delegatingSecurityContextRepository)
      .and()
      .addFilterBefore(licenseValidationFilter, classOf[WebAsyncManagerIntegrationFilter])
      .addFilterAfter(new TaskWorkerAuthenticationFilter(authenticationManager, workerRepository), classOf[UsernamePasswordAuthenticationFilter])
      .addFilterBefore(new LogbackAccessSecurityAttributesSaveFilter(), classOf[TaskWorkerAuthenticationFilter])
      .addFilterAfter(new BasicAuthWithRememberMeFilter(authenticationManager, delegatingAuthenticationEntryPoint), classOf[TaskWorkerAuthenticationFilter])
      .addFilterAfter(new LicenseExpiryCheckFilter(licenseService, daysBeforeWarning), classOf[TaskWorkerAuthenticationFilter])
      .exceptionHandling()
      .authenticationEntryPoint(delegatingAuthenticationEntryPoint)
      .and()
      .formLogin()
      .failureHandler(authenticationFailureHandler)
      .successHandler(new WithoutRedirectLoginSuccessHandler())
      .loginPage("/login")
      .and()
      .rememberMe().key(rememberMeKey).rememberMeServices(rememberMeServices).and()
      .logout().logoutSuccessHandler(withoutRedirectLogoutSuccessHandler).and()

      .sessionManagement().sessionCreationPolicy(SessionCreationPolicy.NEVER).and()

      .requestCache().requestCache(new NullRequestCache()).and()
      .exceptionHandling().accessDeniedHandler(accessDeniedHandler).and()

      .authorizeRequests()
      .antMatchers(s"/$prefix/auth").permitAll()
      .antMatchers("/centralConfiguration/**").hasRole(ServerConfigurationHelper.ConfigRole)
      .antMatchers("/actuator/**").hasRole("ADMIN")
      .antMatchers("/v1/roles/**").denyAll()
      .antMatchers(s"/$prefix/**", "/ws").fullyAuthenticated()
      .accessDecisionManager(accessDecisionManager).and()

      .sessionManagement().maximumSessions(serverSideConfiguration.session.maximumSessions).maxSessionsPreventsLogin(serverSideConfiguration.session.exceptionIfMaximumExceeded)

    logger.info("Starting url prefix '{}' with default security config", prefix)
  }
}
