package com.xebialabs.deployit.provision.steps

import com.xebialabs.deployit.plugin.api.deployment.specification.Operation
import com.xebialabs.deployit.plugin.api.flow.{ExecutionContext, Preview, PreviewStep, StepExitCode}
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.rules.{RulePostConstruct, Scope, StepMetadata, StepPostConstructContext}
import com.xebialabs.deployit.plugin.api.udm._
import com.xebialabs.deployit.provision.ProvisionHelper._
import com.xebialabs.deployit.provision.{BaseDeployedInfrastructureAsCodeType, ProvisionHelper}
import com.xebialabs.deployit.repository.{RepositoryService, RepositoryServiceHolder}

import scala.collection.JavaConverters._

@StepMetadata(name = "create-dictionary")
class CreateDictionaryStep extends BaseStep with PreviewStep with ScopeValidator {

  private var environmentId: String = _
  private var deployed: BaseDeployedInfrastructureAsCodeType = _
  private var addEnvironmentToBoundCIs = false

  override def getPreview: Preview = {
    Preview.withContents(s"Create dictionary '$getDictionaryId'")
  }

  @RulePostConstruct
  def validateAndSetParameters(ctx: StepPostConstructContext): Unit = {
    validate(ctx)
    order = if (order != null) order else 80
    deployed = ctx.getDelta.getDeployed.asInstanceOf[BaseDeployedInfrastructureAsCodeType]
    environmentId = getProvisionEnvironmentId(deployed.getEnvironmentPath, ctx.getDeployedApplication.getEnvironment.getId)
    addEnvironmentToBoundCIs = environmentId != ctx.getDeployedApplication.getEnvironment.getId
    if (description == null || description.isEmpty) {
      description = s"Create dictionary for ${deployed.getName} from output"
    }
  }

  private def validate(ctx: StepPostConstructContext): Unit = {
    validateScope(ctx.getScope, Scope.DEPLOYED)
    if (!ctx.getDelta.getDeployed.isInstanceOf[BaseDeployedInfrastructureAsCodeType]) {
      throw new IllegalArgumentException(s"This step can only be applied to '${Type.valueOf(classOf[BaseDeployedInfrastructureAsCodeType]).toString}' Configuration Items")
    }
    if (ctx.getDelta.getOperation != Operation.CREATE && ctx.getDelta.getOperation != Operation.MODIFY) {
      throw new IllegalArgumentException(s"This step can only be applied to '${Operation.CREATE}' or '${Operation.MODIFY}' operations")
    }
  }

  protected def repository: RepositoryService = {
    RepositoryServiceHolder.getRepositoryService
  }

  override def execute(ctx: ExecutionContext): StepExitCode = {
    implicit val context: ExecutionContext = ctx
    val dictionaryId = getDictionaryId
    val dictionary = getDictionary(dictionaryId)
    dictionary.setEntries(deployed.getOutputVariables)
    repository.createOrUpdate(dictionary)
    context.logOutput(s"Dictionary created: '$dictionaryId'")
    val environment = getEnvironment
    if (!environment.getDictionaries.asScala.map(_.getId).contains(dictionaryId)) {
      environment.getDictionaries.add(dictionary)
      repository.update(environment)
      context.logOutput(s"Dictionary added to Environment '$environmentId'")
    }
    deployed.getBoundConfigurationItems.add(dictionary)
    if (addEnvironmentToBoundCIs) {
      deployed.getBoundConfigurationItems.add(environment)
    }
    StepExitCode.SUCCESS
  }

  private def getDictionary(dictionaryId: String)(implicit context: ExecutionContext): Dictionary =
    if (repository.exists(dictionaryId)) {
      val ci: ConfigurationItem = repository.read(dictionaryId)
      if (!ci.isInstanceOf[Dictionary]) {
        throw new IllegalArgumentException(s"'$dictionaryId' is not a dictionary")
      }
      ci.asInstanceOf[Dictionary]
    } else {
      createDirectories(dictionaryId, context)
      val dict = new Dictionary()
      dict.setId(dictionaryId)
      dict
    }

  private def getDictionaryId: String = {
    Option(deployed.getDictionaryPath) match {
      case Some(s) if !s.isEmpty =>
        generateId(deployed.getDictionaryPath, environmentId)
      case _ => environmentId.substring(0, environmentId.lastIndexOf("/") + 1) + deployed.getName + "-dictionary"
    }
  }

  private def getEnvironment(implicit context: ExecutionContext): Environment = getOrCreateEnvironment(environmentId, context)

}
