package com.xebialabs.deployit.provision
package steps

import com.xebialabs.deployit.plugin.api.flow.{ExecutionContext, Preview, PreviewStep, StepExitCode}
import com.xebialabs.deployit.plugin.api.rules.{RulePostConstruct, Scope, StepMetadata, StepPostConstructContext}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.repository.ChangeSet
import com.xebialabs.deployit.util.Tuple
import grizzled.slf4j.Logging

import scala.collection.convert.ImplicitConversions._

@StepMetadata(name = "delete-all-provisioned-cis")
class DeleteAllProvisionedItems extends ProvisionedItemsStep with PreviewStep with Logging {

  override def validScope: Scope = Scope.PRE_PLAN

  @RulePostConstruct
  def validateAndSetParameters(ctx: StepPostConstructContext) {
    validate(ctx)
    if (description == null || description.isEmpty) {
      description = s"Delete all provisioned configuration items"
    }
    order = if (order != null) order else 25
    setParameters(ctx)
  }

  override def getPreview: Preview = {
    Preview.withContents(s"Deleting following configuration items:\n${allCIsToDelete.mkString("\n")}")
  }

  override def execute(ctx: ExecutionContext): StepExitCode = {
    ctx.logOutput("Starting deletion of provisioned items")
    val changeSet = new ChangeSet
    val ciIds = allCIsToDelete
    ctx.logOutputRaw(s"Deleting following configuration items:\n${ciIds.mkString("\n")}\n")
    changeSet.delete(ciIds.toList.map(Tuple.of(null: Integer, _)))
    resetProvisionedEnvironment(changeSet)
    resetDeployeds(changeSet)
    repositoryService.execute(changeSet)
    ctx.logOutput("Deletion of provisioned items is done")
    StepExitCode.SUCCESS
  }

  private def allCIsToDelete: Set[String] = {
    resolvedTemplateOnDeployedApplication ++ resolvedTemplatesOnDeployeds ++ generatedConfigurationItems
  }

  private def resolvedTemplateOnDeployedApplication: Set[String] = {
    deployedApplication.getBoundConfigurationItems.map(_.getId).toSet
  }

  private def resolvedTemplatesOnDeployeds: Set[String] = {
    deployeds.flatMap(_.getBoundConfigurationItems).map(_.getId)
  }

  private def generatedConfigurationItems: Set[String] = {
    deployedInfrastructureAsCode
      .flatMap(_.getGeneratedConfigurationItems)
      .map(_.getId)
  }

  private def deployedInfrastructureAsCode: Set[BaseDeployedInfrastructureAsCodeType] =
    deployeds
      .filter(_.isInstanceOf[BaseDeployedInfrastructureAsCodeType])
      .map(_.asInstanceOf[BaseDeployedInfrastructureAsCodeType])

  private def resetDeployeds(changeSet: ChangeSet): Unit = {
    val deployedWithBoundCIs = deployeds.filterNot(_.getBoundConfigurationItems.isEmpty)
    deployedWithBoundCIs.foreach(_.setBoundConfigurationItems(new JHashSet[ConfigurationItem]))
    deployedInfrastructureAsCode.foreach(_.setGeneratedConfigurationItems(new JHashSet[ConfigurationItem]))
    changeSet.update(deployedWithBoundCIs.toList)
    changeSet.update(deployedInfrastructureAsCode.toList)
  }

  private def resetProvisionedEnvironment(changeSet: ChangeSet): Unit = {
    if (repositoryService.exists(deployedApplication.getId)) {
      val provisionedEnvironment = getEnvironment
      provisionedEnvironment.setMembers(provisionedEnvironment.getMembers.filter(member => !allCIsToDelete.contains(member.getId)))
      provisionedEnvironment.setDictionaries(provisionedEnvironment.getDictionaries.filter(dictionary => !allCIsToDelete.contains(dictionary.getId)))
      changeSet.update(provisionedEnvironment)
      deployedApplication.setBoundConfigurationItems(new JHashSet[ConfigurationItem]())
      deployedApplication.setDeployeds(deployedApplication.getDeployeds.filter(ci => repositoryService.exists(ci.getId)))
      changeSet.update(deployedApplication)
    }
  }

}
