package com.xebialabs.deployit.provision
package resolver.placeholder

import com.samskivert.mustache.Escapers._
import com.samskivert.mustache.Mustache
import com.xebialabs.deployit.service.replacement.{ConsolidatedDictionary, MustachePlaceholderScanner}
import grizzled.slf4j.Logging

import scala.jdk.CollectionConverters._

private[resolver] class MustachePlaceholderResolver(dictionary: ConsolidatedDictionary,
                                                    sourceDeployed: Option[DeployedType]) extends PlaceholderResolver with PlaceholderEncloser with Logging {

  val scanner = new MustachePlaceholderScanner()

  override def scanPlaceholdersAndReplace(propertyValue: String,
                                          ignoreContextPlaceholders: Boolean = false,
                                          delimiters: String): String = try {
    val map: JMap[String, String] = scanner
      .scan(propertyValue)
      .asScala
      .map(placeholder => placeholder -> fetchValue(placeholder))
      .toMap[String, String]
      .asJava
    Mustache.compiler().withEscaper(NONE).compile(propertyValue).execute(map)
  } catch {
    case e: Exception =>
      if (ignoreContextPlaceholders && propertyValue.contains(contextDelimiter)) {
        logger.warn(s"Placeholder values could not be resolved for input '$propertyValue'")
        propertyValue
      } else {
        throw new IllegalArgumentException(s"Placeholder values could not be resolved for input '$propertyValue'", e)
      }
  }

  def getValueForMapKey(placeholder: String, d: DeployedType): String = {
    val dotIndex = placeholder.indexOf('.')
    val propertyName = placeholder.substring(1, dotIndex)
    val keyName = placeholder.substring(dotIndex + 1, placeholder.length - 1)
    if (d.getProperty(propertyName).isInstanceOf[JMap[String, String]]) {
      val placeHolderValue: JMap[String, String] = d.getProperty(propertyName)
      if (!placeHolderValue.containsKey(keyName)) {
        throw new IllegalArgumentException(s"Placeholder key $keyName does not exists in property $propertyName.")
      }
      placeHolderValue.get(keyName)
    } else {
      throw new IllegalArgumentException("Contextual placeholder delimiter %map.key% can only be used for map_string_string properties.")
    }
  }

  private[this] def fetchValue(placeholder: String): String = {
    sourceDeployed match {
      case Some(d) if isEnclosedBy(placeholder, contextDelimiter) && placeholder.contains('.') => getValueForMapKey(placeholder, d)
      case Some(d) if isEnclosedBy(placeholder, contextDelimiter) => d.getProperty(stripEnclosingDelimiters(placeholder)).toString
      case _ if isEnclosedBy(placeholder, literalDelimiter) => "{{" + stripEnclosingDelimiters(placeholder) + "}}"
      case _ => dictionary.get(placeholder)
    }
  }
}

object MustachePlaceholderResolver {

  def apply(dictionary: ConsolidatedDictionary, sourceDeployed: DeployedType): MustachePlaceholderResolver = {
    new MustachePlaceholderResolver(dictionary, Option(sourceDeployed))
  }

}

