package com.xebialabs.deployit.provision

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.{ConfigurationItem, Environment}
import com.xebialabs.deployit.repository.core.Directory
import com.xebialabs.deployit.repository.{RepositoryService, RepositoryServiceHolder}

object ProvisionHelper {

  def getProvisionEnvironmentId(environmentPath: String, targetEnvironmentId: String): String = {
    Option(environmentPath) match {
      case Some(s) if s.nonEmpty =>
        generateId(environmentPath, targetEnvironmentId)
      case _ => targetEnvironmentId
    }
  }

  def generateId(newPath: String, targetEnvironmentId: String): String = {
    var path = newPath
    if (path.endsWith("/")) {
      path = path.dropRight(1)
    }
    if (path.startsWith("/")) {
      path = path.drop(0)
    }
    if (path.startsWith("Environments/")) {
      path
    } else {
      targetEnvironmentId.substring(0, targetEnvironmentId.lastIndexOf("/") + 1) + path
    }
  }

  def getOrCreateEnvironment(environmentId: String, context: ExecutionContext): Environment = {
    if (repository.exists(environmentId)) {
      readEnvironment(environmentId)
    } else {
      createDirectories(environmentId, context)
      val environment = new Environment()
      environment.setId(environmentId)
      repository.create(environment)
      context.logOutput(s"Environment created: '$environmentId'")
      repository.read(environmentId)
    }
  }

  def createDirectories(fullId: String, context: ExecutionContext): Unit = {
    "/".r.findAllMatchIn(fullId).foreach{ index =>
      if (index.start > 0) {
        val id = fullId.substring(0, index.start)
        if (repository.exists(id)){
          val ci: ConfigurationItem = repository.read(id)
          if (!ci.getType.instanceOf(Type.valueOf("internal.Root")) &&
            !ci.getType.instanceOf(Type.valueOf("core.Directory"))) {
            throw new IllegalArgumentException(s"'$fullId' is invalid. '$id' must be a directory under 'Environments'.")
          }
        } else {
          val directory = new Directory()
          directory.setId(id)
          repository.create(directory)
          context.logOutput(s"Directory created: '$id'")
        }
      }
    }
  }

  def readEnvironment(environmentId: String): Environment = {
    val ci: ConfigurationItem = repository.read(environmentId)
    if (!ci.isInstanceOf[Environment]) {
      throw new IllegalArgumentException(s"'$environmentId' is not an environment")
    }
    ci.asInstanceOf[Environment]
  }

  protected def repository: RepositoryService = {
    RepositoryServiceHolder.getRepositoryService
  }

}
