
import json
from terraform.mapper.gke_cluster_mapper import GKEClusterMapper
from terraform.mapper.aks_cluster_mapper import AKSClusterMapper
from com.xebialabs.deployit.provision import ProvisionHelper
from com.xebialabs.overthere.util import OverthereUtils

def capture_state(deployed):
    encoding="UTF-8"
    connection = deployed.container.host._delegate.getConnection()
    state_file = "{}{}{}{}terraform.tfstate".format(deployed.container.workingDirectory, deployed.container.host.os.fileSeparator,deployed.name,deployed.container.host.os.fileSeparator)
    otfile = connection.getFile(state_file)

    if not otfile.exists():
        state_file = "{}{}{}{}{}{}terraform.tfstate".format(deployed.container.workingDirectory, deployed.container.host.os.fileSeparator,deployed.name,deployed.container.host.os.fileSeparator, ".terraform", deployed.container.host.os.fileSeparator)
        otfile = connection.getFile(state_file)
        if not otfile.exists():
            raise Exception("File [%s] does not exist" % state_file)

    print("Read the state file {0}".format(state_file))

    output = OverthereUtils.read(otfile, encoding)
    output_json = json.loads(output)    
    return output_json

class CreateResources(object):
    def __init__(self, task_context):
        self.repository = task_context['repositoryService']
        self.context = task_context['context']
        self.previousDeployed = task_context['previousDeployed']
        self.deployed = task_context['deployed']
        self.deployedApplication = task_context['deployedApplication']
        self.environment_id = ProvisionHelper.getProvisionEnvironmentId(
            self.deployed.environmentPath, self.deployedApplication.environment.id)
        self.environment = ProvisionHelper.getOrCreateEnvironment(
            self.environment_id, context)
        host_id = deployed.container.host.id
        self.folder = host_id[:host_id.rfind('/')+1]
        self.generated_ids = []
        self.generated_cis = []
        self.cis_to_delete = []
        self.resource_mappers = {
            'google_container_cluster': GKEClusterMapper(),
            'azurerm_kubernetes_cluster': AKSClusterMapper()
        }

    def process(self):
        self.process_resources()
        self.process_cis_to_delete()
        self.update_environment_members()
        self.update_generated_cis()
        self.delete_removed_resources()

    def process_resources(self):        
        output = capture_state(deployed)
        if output:
            output_json = output
            if 'modules' in output_json:
                for module in output_json['modules']:
                    if 'resources' in module:
                        for resourceKey in module['resources']:
                            resource = module['resources'][resourceKey]
                            self.process_resource(resource)
            elif 'resources' in output_json:
                for resource in output_json['resources']:
                    for instance in resource['instances']:
                        instance['type'] = resource['type']
                        self.process_resource(instance)
        else:
            context.logOutput("No resources found for '%s', skipping Infrastructure creation." % deployed.name)

    def process_resource(self, resource):
        if resource['type'] in self.resource_mappers:
            cis = self.resource_mappers[resource['type']].create_ci(
                resource, self.folder)
            for ci in cis:
                if ci is not None:
                    if self.repository.exists(ci.id):
                        self.repository.update(ci.id, ci)
                        print("'%s' of type '%s' updated from '%s' resource." % (
                            ci.id, ci.type, resource['type']))
                    else:
                        self.repository.create(ci.id, ci)
                        print("'%s' of type '%s' created from '%s' resource." % (
                            ci.id, ci.type, resource['type']))
                    self.generated_cis.append(ci)
                    self.generated_ids.append(ci.id)
        else:
            context.logOutput("Skipping '%s' as it is not a candidate for infrastructure creation." % resource['type'])

    def process_cis_to_delete(self):
        if (self.previousDeployed):
            for ci in self.previousDeployed.generatedConfigurationItems:
                if ci.type != "udm.Environment" and ci.type != "udm.Dictionary":
                    if ci.id not in self.generated_ids:
                        self.cis_to_delete.append(ci.id)

    def update_environment_members(self):
        environment = ProvisionHelper.getOrCreateEnvironment(
            self.environment_id, self.context)
        members = environment.members
        for ci in self.generated_cis:
            if 'Environments/' not in ci.id:
                if 'kube-system' in ci.name:
                    self.create_kube_system_environment(ci)
                else:
                    members.add(ci)
        members_to_remove = []
        for ci in members:
            if ci.id in self.cis_to_delete:
                print("'%s' removed from 'members' property of environment '%s'" %
                      (ci.id, self.environment_id))
                members_to_remove.append(ci)
        for ci in members_to_remove:
            members.remove(ci)
        environment.setMembers(members)
        self.repository.update(self.environment_id, environment)

    def update_generated_cis(self):
        generatedConfigurationItems = self.deployed.generatedConfigurationItems
        for ci in self.generated_cis:
            generatedConfigurationItems.add(ci)
        if self.environment_id != self.deployedApplication.environment.id:
            generatedConfigurationItems.add(
                self.repository.read(self.environment_id))

        generated_to_remove = []
        for ci in generatedConfigurationItems:
            if ci.id in self.cis_to_delete:
                print("'%s' removed from 'generatedConfigurationItems' property of '%s'" % (
                    ci.id, self.deployed.id))
                generated_to_remove.append(ci)
        for ci in generated_to_remove:
            generatedConfigurationItems.remove(ci)
        self.deployed.setGeneratedConfigurationItems(
            generatedConfigurationItems)
        if self.repository.exists(self.deployed.id):
            self.repository.update(self.deployed.id, self.deployed)

    def delete_removed_resources(self):
        for ci_id in self.cis_to_delete:
            self.repository.delete(ci_id)
            print("'%s' deleted" % ci_id)


CreateResources(locals()).process()
