#
# Copyright (c) 2021. All rights reserved.
# This software and all trademarks, trade names, and logos included herein are the property of Digital.ai, Inc. and its affiliates, subsidiaries, and licensors.
#

"""
Module for Terraform Enterprise API Endpoint: Workspaces.
"""

import json
import requests

from .endpoint import TFEEndpoint


class TFEWorkspaces(TFEEndpoint):
    """
    Workspaces represent running infrastructure managed by Terraform.

    https://www.terraform.io/docs/enterprise/api/workspaces.html
    """

    def __init__(self, base_url, organization, headers):
        super(TFEWorkspaces, self).__init__(base_url, organization, headers)
        self._ws_base_url = "{base_url}/workspaces".format(base_url=base_url)
        self._org_base_url = "{base_url}/organizations/{organization_name}/workspaces".format(base_url=base_url, organization_name=self._organization_name)

    def create(self, workspace_name, terraform_version="0.12.9"):
        """
        POST /organizations/:organization_name/workspaces
        """
        payload = {
            "data": {
                "attributes": {
                    "name": workspace_name,
                    "auto-apply": "true",
                    "terraform-version": terraform_version,
                    "source-name": "XLDeploy",
                    "source-url": "http://localhost:4516"
                },
                "type": "workspaces"
            }
        }
        return self._create(self._org_base_url, payload)

    def destroy(self, workspace_id=None, workspace_name=None):
        """
        GET /organizations/:organization_name/workspaces/:name
        DELETE /workspaces/:workspace_id

        A workspace can be deleted via two endpoints, which behave identically. One refers to a
        workspace by its ID, and the other by its name and organization.
        """
        if workspace_name is not None:
            url = "{0}/{1}".format(self._org_base_url, workspace_name)
        elif workspace_id is not None:
            url = "{0}/{1}".format(self._ws_base_url, workspace_id)
        else:
            self._logger.error("Arguments workspace_name or workspace_id must be defined")

        return self._destroy(url)

    def show(self, workspace_name=None, workspace_id=None):
        """
        GET /organizations/:organization_name/workspaces/:name
        GET /workspaces/:workspace_id

        Details on a workspace can be retrieved from two endpoints, which behave identically.
        One refers to a workspace by its ID, and the other by its name and organization.
        """
        if workspace_name is not None:
            url = "{0}/{1}".format(self._org_base_url, workspace_name)
        elif workspace_id is not None:
            url = "{0}/{1}".format(self._ws_base_url, workspace_id)
        else:
            self._logger.error("Arguments workspace_name or workspace_id must be defined")

        return self._show(url)

    def get_id(self, workspace):
        workspace_name = self.get_workspace_name(workspace)
        workspace_info = self.show(workspace_name=workspace_name)
        if 'data' in workspace_info:
            ws_id = workspace_info["data"]["id"]
            self._logger.debug("workspace {0} -> id {1}".format(workspace_name, ws_id))
            return ws_id
        else:
            error_msg = "Cannot get workspace id of '{0}':{1}'".format(workspace_name, self.format_error_message(workspace_info))
            self._logger.error(error_msg)
            raise Exception(error_msg)

    def get_workspace_name(self, workspace):
        if workspace.workspaceName is not None:
            return workspace.workspaceName
        return workspace.name

    def lst(self):
        """
        GET /organizations/:organization_name/workspaces

        This endpoint lists workspaces in the organization.
        """
        return self._ls(self._org_base_url)
