#
# Copyright (c) 2021. All rights reserved.
# This software and all trademarks, trade names, and logos included herein are the property of Digital.ai, Inc. and its affiliates, subsidiaries, and licensors.
#

"""
Module for container class of all TFE endpoints and high level exceptions around
API access.
"""

from .workspaces import TFEWorkspaces
from .organizations import TFEOrganizations
from .config_versions import TFEConfigVersions
from .variables import TFEVariables
from .runs import TFERuns
from .state_versions import TFEStateVersions
from .hclparser import HclParser


class InvalidTFETokenException(Exception):
    """Cannot instantiate TFE Api class without a valid TFE_TOKEN."""


class TFE():
    """
    Super class for access to all TFE Endpoints.
    """

    def __init__(self, organization, debug=False):
        self.organization = organization
        if self.organization.token is None:
            raise InvalidTFETokenException

        TFE.configure_logger_stdout(defaultLogLevel=self.organization.defaultLogLevel, debug=debug)

        self._instance_url = "{url}/api/v2".format(url=self.organization.url)
        self._token = self.organization.token
        self._current_organization = None
        self._headers = {
            "Authorization": "Bearer {0}".format(self._token),
            "Content-Type": "application/vnd.api+json"
        }
        self.config_versions = None
        self.variables = None
        self.runs = None
        self.state_versions = None

        if self.organization.organizationName is not None:
            self._set_organization(self.organization.organizationName)
        else:
            self._set_organization(self.organization.name)

        self.hcl_parser = HclParser()

    def _set_organization(self, organization_name):
        """
        Sets the organization to use for org specific endpoints.
        This method must be called for their respective endpoints to work.
        """
        self._current_organization = organization_name
        self.workspaces = TFEWorkspaces(self._instance_url, self.organization, self._headers)
        self.config_versions = TFEConfigVersions(self._instance_url, self.organization, self._headers)
        self.variables = TFEVariables(self._instance_url, self.organization, self._headers)
        self.runs = TFERuns(self._instance_url, self.organization, self._headers)
        self.state_versions = TFEStateVersions(self._instance_url, self.organization, self._headers)
        self.organizations = TFEOrganizations(self._instance_url, self.organization, self._headers)

    log_handler = None

    @staticmethod
    def configure_logger_stdout(defaultLogLevel, debug):
        import logging
        import sys
        root = logging.getLogger()
        if TFE.log_handler == None:
            handler = logging.StreamHandler(sys.stdout)
            TFE.log_handler = handler
            formatter = logging.Formatter('%(asctime)s - %(name)s - %(levelname)s - %(message)s')
            handler.setFormatter(formatter)
            root.addHandler(handler)
        else:
            handler = TFE.log_handler

        if debug:
            root.setLevel(logging.DEBUG)
            handler.setLevel(logging.DEBUG)
        else:
            root.setLevel(defaultLogLevel)
            handler.setLevel(defaultLogLevel)

    def load_variables_in_workspace(self, input, workspace, secret, scope):
        ws_id = self.workspaces.get_id(workspace)
        data = self.variables.lst(workspace_name=self._get_workspace_name(workspace))
        existing_variables = {}
        for variable_info in data['data']:
            existing_variables[variable_info['attributes']['key']] = variable_info['id']

        for key in input:
            value = input[key]
            is_hcl = self.hcl_parser.is_hcl_variable(key)
            if key in existing_variables:
                print("update terraform variable {0} -> {1} ({2})".format(key, self.display_value(value, secret), is_hcl))
                if secret:
                    self.variables.destroy(existing_variables[key])
                    self.variables.create(ws_id, key, value, scope, secret, is_hcl)
                else:
                    self.variables.update(existing_variables[key], key, value, scope, secret, is_hcl)
            else:
                print("new terraform variable {0} -> {1} ({2})".format(key, self.display_value(value, secret), is_hcl))
                self.variables.create(ws_id, key, value, scope, secret, is_hcl)

    def display_value(self, value, secret):
        if secret:
            return 'xxxxxxxxxxxxxx'
        else:
            return value

    def _get_workspace_name(self, workspace):
        if workspace.workspaceName is not None:
            return workspace.workspaceName
        return workspace.name
