package com.xebialabs.xldeploy.provisioner

import java.util.Optional

import com.xebialabs.deployit.booter.local.generator.TypeGenerator
import com.xebialabs.deployit.booter.local.{LocalDescriptor, LocalPropertyDescriptor, TypeDefinitions}
import com.xebialabs.deployit.plugin.api.reflect.{PropertyKind, Type}
import com.xebialabs.deployit.plugin.api.udm.{ConfigurationItem, Deployable, EmbeddedDeployable}
import com.xebialabs.deployit.plugin.api.validation.Validator

import scala.collection.convert.wrapAll._

class TemplateTypeGenerator extends TypeGenerator {

  override def getName: String = "provision-templates"

  override def dependsOn(): JList[String] = Nil

  override def generateAndRegister(typeDefinitions: TypeDefinitions): Unit = {
    typeDefinitions.getDefinitions.filter(td => shouldHaveATemplate(td.getType)).foreach { td =>
      val templateTypeDefinition = new typeDefinitions.TypeDefinition {
        setType(toTemplateType(td.getType))
        setSuperType(Type.valueOf(classOf[Template]))
        val superTypes = (td.getType.getDescriptor.getSuperClasses ++ td.getType.getDescriptor.getInterfaces)
          .filter(shouldHaveATemplate).map(toTemplateType).toSet

        override def createDescriptor(typeDefinitions: TypeDefinitions): LocalDescriptor = {
          new TemplateLocalDescriptor(td.getType.getDescriptor.asInstanceOf[LocalDescriptor], superTypes)
        }

        override def registerAsSubtype(ciType: Type): Unit = {
          super.registerAsSubtype(ciType)
          superTypes.foreach(registerSuperType(_, getType))
        }
      }
      typeDefinitions.addTypeDef(templateTypeDefinition)
    }
  }

  class TemplateLocalDescriptor(localDescriptor: LocalDescriptor, interfaceTypes: Set[Type]) extends LocalDescriptor {
    addSuperClass(Type.valueOf(classOf[Template]))
    setType(toTemplateType(localDescriptor.getType))
    setRootName(Optional.empty())
    setClazz(classOf[Template])
    setDescription(s"Template of ${localDescriptor.getType}")
    setVirtual(localDescriptor.isVirtual)
    initHierarchy()
    interfaceTypes.foreach(addInterface)
    localDescriptor.getInterfaces.filter(shouldHaveATemplate).map(toTemplateType).foreach(addInterface)
    localDescriptor.getSuperClasses.filter(shouldHaveATemplate).map(toTemplateType).foreach(addSuperClass)
    addPropertyDescriptorsFrom(localDescriptor)

    private def addPropertyDescriptorsFrom(descriptor: LocalDescriptor) {
      descriptor.getPropertyDescriptors.foreach(pd =>
        addPropertyDescriptor(new TemplatePropertyDescriptor(this, pd.asInstanceOf[LocalPropertyDescriptor])))
    }

    class TemplatePropertyDescriptor(templateLocalDescriptor: TemplateLocalDescriptor, propertyDescriptor: LocalPropertyDescriptor) extends LocalPropertyDescriptor {
      setDeclaringDescriptor(templateLocalDescriptor)
      setFromPropertyDescriptor(propertyDescriptor)
      addDefaultValidationRules()
      registerDefault(propertyDescriptor)
      reInitializeRequired()

      override def doSetValue(item: ConfigurationItem, value: scala.Any): Unit = {
        templateLocalDescriptor.setSyntheticPropertyValue(item, getName, value)
      }

      override def get(item: ConfigurationItem): AnyRef = {
        templateLocalDescriptor.getSyntheticPropertyValue(item, getName)
      }

      override def setEnumValues(enumValues: JList[String]): Unit = super.setEnumValues(null)

      override def setEnumClass(enumClass: Class[_]): Unit = super.setEnumClass(null)

      override def setInspectionProperty(inspectionProperty: Boolean): Unit = super.setInspectionProperty(false)

      override def setRequiredForInspection(requiredForInspection: Boolean): Unit = super.setRequiredForInspection(false)

      override def setKind(kind: PropertyKind): Unit = {
        if (kind.isSimple) {
          super.setKind(PropertyKind.STRING)
        } else {
          super.setKind(kind)
        }
      }

      override def setReferencedType(referencedType: Type): Unit = if (referencedType != null) {
        if (shouldHaveATemplate(referencedType)) {
          super.setReferencedType(toTemplateType(referencedType))
        } else {
          super.setReferencedType(referencedType)
        }
      }

      override def setValidationRules(validationRules: JSet[Validator[_]]): Unit = {
        super.setValidationRules(new JHashSet())
      }
    }

  }

  private def shouldHaveATemplate(ciType: Type): Boolean = {
    ciType.getPrefix != "upm" &&
      !ciType.isSubTypeOf(typeOf[Provider]) &&
      !ciType.isSubTypeOf(typeOf[Provisioner]) &&
      !ciType.isSubTypeOf(typeOf[Provisionable]) &&
      !ciType.isSubTypeOf(typeOf[EmbeddedDeployedContainerType]) &&
      !ciType.isSubTypeOf(typeOf[EmbeddedDeployable]) &&
      !ciType.isSubTypeOf(typeOf[Deployable])
  }

  private def toTemplateType(ciType: Type): Type = Type.valueOf("template." + ciType.getPrefix, ciType.getName)

}
