#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#

from collections import deque
from sys import exit
from sys import stderr

GRAY, BLACK = 0, 1

def topological(graph):
    order, enter, state = deque(), set(graph), {}

    def dfs(node):
        state[node] = GRAY
        for k in graph.get(node, ()):
            sk = state.get(k, None)
            if sk == GRAY: raise ValueError("Error: cyclic dependency between containers")
            if sk == BLACK: continue
            enter.discard(k)
            dfs(k)
        order.appendleft(node)
        state[node] = BLACK

    while enter: dfs(enter.pop())
    return order

def created_modified_deployeds():
    deployed_set = set()
    for delta in deltas.deltas:
        if (delta.operation == "CREATE" or delta.operation == "MODIFY") and delta.deployedOrPrevious.type == "podman.Container":
            deployed_set.add(delta)
    return deployed_set

def sort_containers(containers):
    graph = {}
    for d in containers:
        if d.deployed.container in graph:
            if (d.deployed.containerName or d.deployed.name) in graph[d.deployed.container]:
                raise Exception("Cannot use container name %s for more than one container deployed to the same Podman engine" % (d.deployed.containerName or d.deployed.name))
        else:
            graph[d.deployed.container] = {}
        graph[d.deployed.container][d.deployed.containerName or d.deployed.name] = d.deployed.links.keys()

    sorted_containers = []
    sorted_engines = graph.keys()
    sorted_engines.sort(key=lambda c: c.name)
    for engine in sorted_engines:
        containers_on_engine = graph[engine]
        for c in reversed(topological(containers_on_engine)):
            f = filter(lambda d: d.deployed.container == engine and (d.deployed.containerName or d.deployed.name) == c, containers)
            sorted_containers.extend(f)

    return sorted_containers

def is_not_only_host_network(networks):
    return len(filter(lambda network: network.lower() == "host", networks)) > 0 and len(networks) > 1

def get_create_plan_title(deployed):
    return "Create Podman container %s and connect to network %s on %s" %(deployed.containerName or deployed.name, deployed.networks[0], deployed.container.name) if len(deployed.networks) > 0 else "Create Podman container %s on %s" %(deployed.containerName or deployed.name, deployed.container.name)

def run_container(delta):
    deployed = delta.deployed

    if deployed.networks and is_not_only_host_network(networks=deployed.networks):
        raise Exception("Host network is added, so cannot add any other network")

    context.addStep(steps.jython(
        description="Pull Podman image %s on %s" % (deployed.image, deployed.container.name),
        order=5,
        script="xldpodman/pull_image.py",
        jython_context={'deployed': deployed}
    ))


    context.addStepWithCheckpoint(steps.jython(
        description=get_create_plan_title(deployed=deployed),
        order=70,
        script="xldpodman/create_container.py",
        jython_context={'deployed': deployed}
    ), delta)

    context.addStep(steps.jython(
        description="Start Podman container %s on %s" % (deployed.containerName or deployed.name, deployed.container.name),
        order=80,
        script="xldpodman/start_container.py",
        jython_context={'deployed': deployed}
    ))


max_showLogsAfter = 0
def find_max_showLogsAfter(delta):
    global max_showLogsAfter
    deployed = delta.deployed
    if deployed.showLogsAfter > max_showLogsAfter:
        max_showLogsAfter = deployed.showLogsAfter

docker_run_containers = created_modified_deployeds()
if len(docker_run_containers) > 0:
    sorted_docker_run_containers = sort_containers(docker_run_containers)
    map(run_container, sorted_docker_run_containers)
    map(find_max_showLogsAfter, sorted_docker_run_containers)
    if max_showLogsAfter:
        context.addStep(steps.wait(
            description="Wait for %d seconds" % max_showLogsAfter,
            order=81,
            seconds=max_showLogsAfter
        ))
