#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#
import textwrap
from pprint import pprint
from xld.kubernetes.core_api_client import KubernetesCoreClient
from xld.kubernetes.factories.handler_factory import ContainerHelperFactory


class DescribeHelper(object):
    def __init__(self, deployed):
        self.__deployed = deployed
        self.__container_helper = ContainerHelperFactory(deployed.container).create()

    def describe_pod(self, name):
        client = KubernetesCoreClient(self.__deployed.container.container)
        pod_response = client.describe_pod(name=name, namespace=self.__container_helper.get_container_name(
            self.__deployed.container))
        if not pod_response:
            raise RuntimeError("Pod not available or ready !")
        else:
            for res in pod_response:
                data = res['data']
                if data:
                    metadata = data.metadata
                    spec = data.spec
                    status = data.status
                    print(textwrap.dedent("""
                        ---------------------------------------
                        Describing Pod {0}
                        ---------------------------------------
                        """.format(res['name'])))
                    # Print pod data
                    cmd = textwrap.dedent("""
                            Name:           {0}
                            Namespace:      {1}
                            Priority:       {2}
                            Node:           {3}/{4}
                            Start Time:     {5}
                            Labels:         {6}
                            Status:         {7}
                            IP:             {8}
                            Containers:
                            """.format(
                        metadata.name, metadata.namespace, spec.priority, spec.node_name, status.host_ip,
                        status.start_time,
                        self.get_labels(metadata.labels), status.phase, status.pod_ip
                    ))

                    for container in spec.containers:
                        container_state = self.get_container_state(container, status)
                        cmd += textwrap.dedent("""
                              - {0}:
                                  Image:          {1}
                                  Ports:          {2}
                                  State:          {3}
                                  Ready:          {4}
                                  Restart Count:  {5}
                                  Limits:         {6}
                                  Requests:       {7}
                                  Environment:    {8}
                                  Mounts:
                                    {9}
                            """.format(
                            container.name, container.image, self.get_ports(container), self.get_state(container_state),
                            self.get_ready_state(container_state), self.get_restart_count_state(container_state),
                            self.get_limits(container.resources.limits), self.get_limits(container.resources.requests),
                            container.env, self.get_volume_mounts(container.volume_mounts)
                        ))

                    cmd += textwrap.dedent("""
                            Conditions:
                        """)

                    for container in status.conditions:
                        cmd += textwrap.dedent("""
                                  -  {0}: {1}
                            """.format(container.type, container.status))

                    cmd += textwrap.dedent("""
                            Volumes:
                        """)

                    for v in spec.volumes:
                        cmd += textwrap.dedent("""
                              - {0}
                                """.format(v.name))

                    cmd += textwrap.dedent("""

                            Events:
                                [Type - Reason](From) : Message
                                -------------------------------
                        """)

                    # Print events
                    events = res['events']
                    for event in events.items:
                        cmd += textwrap.dedent("""
                            -   [{0} - {1}]({2}): {3} 
                            """.format(event.type, event.reason, event.source.component, event.message))
                    print(cmd)
                    print("\n")
                    # Print container logs
                    if res['logs']:
                        print(textwrap.dedent("""
                                ---------------------------------------
                                Container logs for pod {0}
                                ---------------------------------------
                                """.format(res['name'])))
                        for log in res['logs']:
                            if log:
                                pprint(log)

    @staticmethod
    def get_container_state(container, status):
        if status:
            if status.container_statuses:
                for cstat in status.container_statuses:
                    if cstat.name == container.name and container.image in cstat.image:
                        return cstat
        return ""

    @staticmethod
    def get_ready_state(container_state):
        out = ""
        if hasattr(container_state, 'ready'):
            if container_state.ready:
                out = "True"
            else:
                out = "False"
        return out

    @staticmethod
    def get_restart_count_state(container_state):
        out = ""
        if hasattr(container_state, 'restart_count'):
            out = container_state.restart_count
        return out

    @staticmethod
    def get_labels(labels):
        out = ""
        if labels:
            for k, v in labels.items():
                if v:
                    out += "{0}: {1}, ".format(k, v)
        return out

    @staticmethod
    def get_ports(container):
        out = ""
        if hasattr(container, 'ports') and container.ports:
            for port in container.ports:
                out += "{0}/{1}, ".format(port.container_port, port.protocol)
        return out

    @staticmethod
    def get_state(container_state):
        out = ""
        if hasattr(container_state, 'state'):
            state = container_state.state
            if state.running and hasattr(state.running, 'reason'):
                out += "running: {0} ({1})\n".format(state.running.reason, state.running.message)
            if state.terminated and hasattr(state.terminated, 'reason'):
                out += "terminated: {0} ({1})\n".format(state.terminated.reason, state.terminated.message)
            if state.waiting and hasattr(state.waiting, 'reason'):
                out += "waiting: {0} ({1})\n".format(state.waiting.reason, state.waiting.message)
        return out

    @staticmethod
    def get_limits(limits):
        out = ""
        if limits:
            for k, v in limits.items():
                if v:
                    out += "{0}: {1}, ".format(k, v)
        return out

    @staticmethod
    def get_volume_mounts(mounts):
        out = ""
        if mounts:
            for port in mounts:
                out += "{0} from {1}\n".format(port.mount_path, port.name)
        return out
