#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#

from abc import abstractmethod
from xld.kubernetes import client
from xld.kubernetes.core_api_client import KubernetesCoreClient
from openshift.dynamic import DynamicClient

class Resource(object):
    def __init__(self, container, api_version='v1'):
        self.k8s_client = KubernetesCoreClient(container.container)
        self.api_version = api_version

    def get_api_client(self, kind):
        print ("[Using API version: {}]".format(self.api_version))
        dynamic_client = DynamicClient(self.k8s_client.internal_api_client)
        return dynamic_client.resources.get(api_version=self.api_version, kind=kind)

    @abstractmethod
    def create(self, namespace, resource_definition):
        pass

    @abstractmethod
    def modify(self, namespace, resource_definition):
        pass

    @abstractmethod
    def delete(self, namespace, resource_definition):
        pass

    @abstractmethod
    def filter_resources_by_definition(self, namespace, resource_definition):
        pass


class DynamicResourceProvider(Resource):
    def __init__(self, container, kind, api_version):
        super(DynamicResourceProvider, self).__init__(container, api_version)
        self.kind=kind
        self.api_client = self.get_api_client(self.kind)
        self.patch_types = {
            "json": "application/json-patch+json",
            "merge": "application/merge-patch+json",
            "strategic": "application/strategic-merge-patch+json"
        }

    def create(self, namespace, resource_definition):
        return self.api_client.create(body=resource_definition, namespace=namespace)

    def modify(self, namespace, resource_definition, patch_type='strategic', update_method='patch'):
        if patch_type not in self.patch_types:
            raise RuntimeError("Merge patch type '{}' not supported. Please use 'json', 'merge' or 'strategic'.".format(patch_type))
        if update_method == 'patch':
            return self.api_client.patch(body=resource_definition, namespace=namespace, content_type=self.patch_types[patch_type])
        else:
            return self.api_client.replace(body=resource_definition, namespace=namespace)

    def delete(self, namespace, resource_definition, propagation_policy='Foreground'):
        return self.api_client.delete(name=resource_definition["metadata"]["name"], namespace=namespace, body=client.V1DeleteOptions(
            propagation_policy=propagation_policy
        ))

    def filter_resources_by_definition(self, namespace, resource_definition):
        return self.api_client.get(namespace=namespace, field_selector="metadata.name={}".format(
            resource_definition["metadata"]["name"]))

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        if self.k8s_client:
            self.k8s_client.__exit__(exc_type, exc_val, exc_tb)

