#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#

from kubernetes import client
from kubernetes.client.rest import ApiException
from xld.kubernetes import KubernetesBaseClient
from xld.kubernetes.factories.handler_factory import ContainerHelperFactory


class KubernetesCoreClient(KubernetesBaseClient):
    def __init__(self, container):
        super(KubernetesCoreClient, self).__init__(container)

    def get_core_api(self):
        version = self.get_api_version()
        print ("[Using API version: {}]".format(self.get_api_version()))
        return self.get_version_client(version)

    def get_api_version(self):
        return "v1"

    def create_pod(self, v1Pod, namespace):
        return self.get_core_api().create_namespaced_pod(namespace=namespace, body=v1Pod)

    def is_pod_status_ready(self, name, container):
        container_helper = ContainerHelperFactory(container).create()
        pod = self.get_core_api().read_namespaced_pod(name=name, namespace=container_helper.get_container_name(container))
        is_ready = bool([condition for condition in pod.status.conditions if condition.type == 'Ready' and condition.status == 'True'])
        print("Pod {0} on {1} {2} is{3} ready.".format(pod.metadata.name, container_helper.get_container_label(),
                                                            container_helper.get_container_name(container), '' if is_ready else ' not'))
        return is_ready

    def detect_crashback_loop_off_error(self, name, container):
        container_helper = ContainerHelperFactory(container).create()
        pod = self.get_core_api().read_namespaced_pod(name=name, namespace=container_helper.get_container_name(container))
        message = [status.state.waiting.message for status in pod.status.container_statuses
                   if hasattr(status.state.waiting, 'reason') and status.state.waiting.reason == 'CrashLoopBackOff']
        return message

    def is_pod_existing(self, name, namespace):
        try:
            pod = self.get_core_api().read_namespaced_pod(name=name, namespace=namespace)
            if pod:
                return True
        except ApiException as e:
            if e.status == 404:
                return False

    def is_namespace_existing(self, namespace):
        try:
            namespace = self.get_core_api().read_namespace(name=namespace)
            if namespace:
                return True
        except ApiException as e:
            if e.status == 404:
                return False

    def remove_pod(self, name, namespace):
        return self.get_core_api().delete_namespaced_pod(name=name, body={}, namespace=namespace)

    def replace_pod(self, name, v1Pod, namespace):
        return self.get_core_api().replace_namespaced_pod(namespace=namespace, body=v1Pod, name=name)

    def create_namespace(self, v1Namespace):
        return self.get_core_api().create_namespace(v1Namespace)

    def remove_namespace(self, v1Namespace):
        return self.get_core_api().delete_namespace(v1Namespace, body=client.V1DeleteOptions(), grace_period_seconds=0)

    def create_service(self, v1Service, namespace):
        return self.get_core_api().create_namespaced_service(namespace=namespace, body=v1Service)

    def is_service_existing(self, name, namespace):
        try:
            service = self.get_core_api().read_namespaced_service(name=name, namespace=namespace)
            if service:
                return True
        except ApiException as e:
            if e.status == 404:
                return False

    def read_namespaced_service(self, name, namespace):
        try:
            service = self.get_core_api().read_namespaced_service(name=name, namespace=namespace)
            if service:
                return service
        except ApiException as e:
            if e.status == 404:
                return False

    def remove_service(self, name, namespace):
        return self.get_core_api().delete_namespaced_service(name=name, namespace=namespace,
                                                             body=client.V1DeleteOptions())

    def replace_service(self, name, v1Service, namespace):
        return self.get_core_api().replace_namespaced_service(namespace=namespace, body=v1Service, name=name)
