#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#

from kubernetes import client


class IngressHelper(object):
    @staticmethod
    def read_ingress(deployed):
        spec = client.V1beta1IngressSpec()
        spec.rules = IngressHelper.__read_rules(deployed.rules)
        if bool(deployed.defaultServiceName) and bool(deployed.defaultServicePort):
            spec.backend = client.V1beta1IngressBackend(service_name=deployed.defaultServiceName,
                                                        service_port=deployed.defaultServicePort)
        if bool(deployed.tlsSet):
            spec.tls = IngressHelper.__read_tls(deployed.tlsSet)
        ingress = client.V1beta1Ingress(spec=spec, metadata=IngressHelper.__read_metadata(deployed))
        return ingress

    @staticmethod
    def __read_metadata(deployed):
        metadata = client.V1ObjectMeta(name=IngressHelper.get_name(deployed))
        if bool(deployed.annotations):
            metadata.annotations = deployed.annotations
        return metadata

    @staticmethod
    def __read_rules(rules):
        result = []
        if bool(rules):
            for rule in rules:
                r = client.V1beta1IngressRule()
                if bool(rule.host):
                    r.host = rule.host
                if bool(rule.paths):
                    r.http = client.V1beta1HTTPIngressRuleValue(paths=IngressHelper.__read_paths(rule.paths))
                result.append(r)
        return result

    @staticmethod
    def __read_paths(paths):
        result = []
        if bool(paths):
            for path in paths:
                p = client.V1beta1HTTPIngressPath(
                    backend=client.V1beta1IngressBackend(service_name=path.serviceName, service_port=path.servicePort))
                if bool(path.path):
                    p.path = path.path
                result.append(p)
        return result

    @staticmethod
    def __read_tls(tlsSet):
        result = []
        if bool(tlsSet):
            for tls in tlsSet:
                t = client.V1beta1IngressTLS(hosts=tls.hosts, secret_name=tls.secretName)
                result.append(t)
        return result

    @staticmethod
    def get_name(deployed):
        return deployed.ingressName if deployed.ingressName else deployed.name
