#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#

from xld.kubernetes.commons.common_utils import CommonUtils
from xld.kubernetes.storage_api_client import KubernetesStorageClient
from xld.kubernetes.storage_class.helper import StorageClassHelper
from kubernetes import client as kubernetes_client
from kubernetes.client.rest import ApiException


class StorageClassService(object):
    def __init__(self, deployed):
        self.__deployed = deployed
        self.__storage_class_helper = StorageClassHelper(self.__deployed)
        self.__kind = CommonUtils.get_kind(self)
        self.__container = deployed.container
        self.__client = KubernetesStorageClient(self.__container)

    def create(self):
        storage_class_request = self.__storage_class_helper.read_storage_class()
        print("Creating {0} '{1}' ".format(self.__kind, StorageClassHelper.get_name(self.__deployed)))
        CommonUtils.print_request_dump(self.__container, storage_class_request, self.__kind,
                                       self.__client.get_api_version())
        storage_class_response = self.__client.get_storage_api().create_storage_class(body=storage_class_request)
        CommonUtils.print_response_dump(self.__container, storage_class_response)
        return storage_class_response

    def modify(self):
        storage_class_request = self.__storage_class_helper.read_storage_class()
        print("Updating {0} '{1}' ".format(self.__kind, StorageClassHelper.get_name(self.__deployed)))
        CommonUtils.print_request_dump(self.__container, storage_class_request, self.__kind,
                                       self.__client.get_api_version())
        storage_class_response = self.__client.get_storage_api().replace_storage_class(
            name=StorageClassHelper.get_name(self.__deployed), body=storage_class_request)
        CommonUtils.print_response_dump(self.__container, storage_class_response)
        return storage_class_response

    def destroy(self):
        print("Destroying {0} '{1}' ".format(self.__kind, StorageClassHelper.get_name(self.__deployed)))
        self.__client.get_storage_api().delete_storage_class(name=StorageClassHelper.get_name(self.__deployed),
                                                             body=kubernetes_client.V1DeleteOptions())

    def is_existing(self, name):
        try:
            storage_class = self.__client.get_storage_api().read_storage_class(name)
            if storage_class:
                return True
        except ApiException as e:
            if e.status == 404:
                return False

    def should_wait_for_creation(self):
        if not self.is_existing(StorageClassHelper.get_name(self.__deployed)):
            print("Waiting for {0} {1} to be created. ".format(self.__kind,
                                                                 StorageClassHelper.get_name(self.__deployed)))
            return True
        return False

    def should_wait_for_destroy(self):
        if self.is_existing(StorageClassHelper.get_name(self.__deployed)):
            print("Waiting for {0} {1} to be destroyed. ".format(self.__kind,
                                                               StorageClassHelper.get_name(self.__deployed)))
            return True
        return False
